// Forth specific routines of of the CForth C011(SO8) project *****************
//
// Author and (c)2025 Wolfgang Schemmert. The copyright claim is restricted
// to this file and the depending files "main.c", "main.h", "cforth.h",
// and "globaldefines.h"
// Other depending material is published, copyrighted and licensed
// by STMicroelectronics and Segger GmbH
//
// Contact: <www.midi-and-more.de>
// A detailled hardware construction and operation manual
// is provided at this website
//
// ***************************************************************************
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//	Licence: GPL-3
//
// ***************************************************************************

#include "stm32c011xx.h"
#include "globaldefines.h"
#include "cforth.h"


void quit(void)   // **************************************************
// ********************************************************************
{
uint32_t k;
const uint8_t ErrorStr [] = "Error:";
const uint8_t ColStr [] = " (col ";
const uint8_t NotFoundStr [] = "_not found";
const uint8_t SyntaxStr [] = "_Syntax";
const uint8_t StackStr [] = "_Stack";
const uint8_t CompoStr [] = "_Compile only";
const uint8_t CodeAccessStr [] = "_Invalid Code access";
const uint8_t InvalContextStr [] = "_out of Context";
const uint8_t RunOnlyStr [] = "_Runtime only";
const uint8_t CancelStr [] = "_Thread cancelled";
const uint8_t RangeStr [] = "_Number out of Range";
const uint8_t Div0Str [] = "_Div.by Zero";
const uint8_t CompilingStr [] = "  compiling..";

	if (gfSP >= gfDataStack + MAXDATASTACK - 4) gfErrors |= ERR_STACK;
	if (gfSP < gfDataStack + 3) gfErrors |= ERR_STACK;

	if (gfErrors != 0) {
		if (gfCompiState) {
		   for (k=0; k<8; k++) {
			   gfUserHeader[gfNextNameCompi].name[k] = 0;
		   }
			   gfUserHeader[gfNextNameCompi].toCompile = 0;
			   gfUserHeader[gfNextNameCompi].attributes = 0;
		}

		gfCompiState = 0;		//terminate compilation in any case

		if (gfTokenStart)  {
			for (k = 0;k < gfTokenStart;k++) sendTB(SPACE);
			sendTB('^');
			sendTB(CRR);
		}
		sendString(ErrorStr);
		if ((gfErrors & ERR_STRINGTOKEN) != 0) sendString(NotFoundStr);
		if ((gfErrors & ERR_SYNTAX) != 0)  sendString(SyntaxStr);
		if ((gfErrors & ERR_STACK) != 0)  sendString(StackStr);
		if ((gfErrors & ERR_COMPONLY) != 0)  sendString(CompoStr);
		if ((gfErrors & ERR_USERCODE) != 0)  sendString(CodeAccessStr);
		if ((gfErrors & ERR_CONTEXT) != 0) sendString(InvalContextStr);
		if ((gfErrors & ERR_CANCEL) != 0)  sendString(CancelStr);
		if ((gfErrors & ERR_RUNTOKEN) != 0)  sendString(RunOnlyStr);
		if ((gfErrors & ERR_NUMVAL) != 0)  sendString(RangeStr);
		if ((gfErrors & ERR_DIVZERO) != 0) sendString(Div0Str);
		sendString(ColStr);
		sendNS32(gfTokenStart);
		sendTB(0x29);  //')'
		op_ABORT();			//resets gfErrors, too
	}
	else  {
		if (gfCompiState) {				 //compiling
			sendString(CompilingStr);
		}
		else if (gfComment != 1) {	//interpreting and no full line comment
			sendTB(SPACE);
			sendTB('o');
			sendTB('k');
			if (gfTempNumBase == 16)  sendTB('H');
		}
	}
	sendTB(CRR);
	gfBackNumValid = 0;
}


uint32_t query(void)   // **************************************************
// *************************************************************************
{
uint32_t tibstate;
uint32_t recvword;
const uint8_t RestartStr [] ="COLD RESET? ";

	gfTibPos = 0;
	gfTibMax = 0;
	gfComment = 0;			//reset and start every line not in comment state
	
	do {
//serve Background process
		if (gfStepSpeed)  {
			if ((gfUserHeader[4].toCompile > 0) ||
								(gfUserHeader[4].toCompile == 0xFFFFFFFE)) {
				if (gfSTEPCOUNT == 0) subMOT();
			}
		}
		tibstate = subQuery();
	} while(tibstate == 0);
	return 1;
}


uint32_t subQuery(void)   // ************************************************
// **************************************************************************
{
uint32_t recvword;
uint8_t recvbyte;

	recvword = pullTermRxBuf();
	if (recvword == 0) return 0;
	recvbyte = (uint8_t)(recvword & 0x000000FF);
	if (gfBackProcess && gfExecUser)  {
		gfTermChar = recvbyte;
		gfTermCheck = 1;
	}
	if (recvbyte == LFF)  return 0;	//ignore LFF
	if (recvbyte <= CTRL_B)  return 0;	//wrong TIB format,ignore and go on

//if not executing User Token, convert TAB -> SPACE
	if (gfExecUser == 0)  {
		if (recvbyte == TAB)  recvbyte = SPACE;
		if ((gfTibPos >= 64) && (recvbyte == SPACE))  recvbyte = CRR;
								//force line break if line is too long
	}

//block loading comments:
	if ((recvbyte == 0x28) && (gfComment == 0)) {	//0x28 = '('
		sendTB(0x28);
		while ((recvword = pullTermRxBuf()) == 0);
		recvbyte = (uint8_t)(recvword & 0x000000FF);
		if (recvbyte == SPACE) {					// ( ..style comment
			gfComment = gfTibPos + 1;
			sendTB(BKSP);
			sendTB(SPACE);
			sendTB(BKSP);
		}
		else  {
			gfTib[gfTibPos] = 0x28;
			gfTibPos++;
			gfTib[gfTibPos] = recvbyte;
			gfTibPos++;
			sendTB(recvbyte);
		}
		return 0;
	}

	if ((recvbyte == 0x2F) && (gfComment == 0)) {	//0x2F = '/'
		sendTB(0x2F);
		while ((recvword = pullTermRxBuf()) == 0);
		recvbyte = (uint8_t)(recvword & 0x000000FF);
		if (recvbyte == 0x2F) {		// '//'	style comment
			gfComment = gfTibPos + 1;
			sendTB(BKSP);
			sendTB(SPACE);
			sendTB(BKSP);
		}
		else  {
			gfTib[gfTibPos] = 0x2F;
			gfTibPos++;
			gfTib[gfTibPos] = recvbyte;
			gfTibPos++;
			sendTB(recvbyte);
		}
		return 0;
	}

	if (recvbyte == ESC)  {
		if (gfExecUser)  {		//-->termination of User Thread
			sendTB(CRR);
			gfTib[gfTibPos] = SPACE;
			gfTib[gfTibPos + 1] = 0;
			gfTibMax = gfTibPos;
			return 2;
		}
		else return 0;
	}

	if (recvbyte == TAB)  return 3;
	if (recvbyte == CTRL_R)  return 4;

	if ((recvbyte == BKSP) && (gfComment == 0)) {
		if (gfTibPos > 0)  {
			gfTibPos--;
			gfTib[gfTibPos]=SPACE;
			sendTB(BKSP);
			sendTB(SPACE);
			sendTB(BKSP);
		}
		return 0;
	}
//avoid line feed after empty comment lines:
   if (recvbyte == CRR)  {
		if (gfTibPos && (gfComment != 1)) sendTB(CRR);
		gfTib[gfTibPos] = SPACE;		//final token terminator for "eval"
		gfTib[gfTibPos + 1] = 0;
		gfTibMax = gfTibPos;
		return 1;
	}
//finally: 	
	if (gfComment == 0)  {
		gfTib[gfTibPos] = (uint8_t)recvbyte;
		sendTB((uint8_t)recvbyte);
		gfTibPos++;
	}
	return 0;
}


void eval(void)   // ******************************************************
//evaluate "normal" terminal input
// ************************************************************************
{
	gfExecUser = 0;
	gfTokenStart = 0;
	gfTibPos = 0;

	while (gfTibPos < gfTibMax)  {
		scanNext();
		if (gfTokenstr[0] == 0) continue;	//ignore CTRL_B,TAB,LFF,ESC	

//return of interpretNumber():1:number, 0:look for token
		if (interpretNumber()) {			//valid number parsed
			if (gfCompiState == 0) {		// interpreting
				gfSP ++;
				*gfSP = gfTokenNum;
			}
			else  {
				if ((uint32_t)gfTokenNum < 256)  { 
					compileTCode(IDX_DOLIT8B);
					compileTCode(gfTokenNum);
				}
				else  {
					compileTCode(IDX_DOLIT);
					compileTCode(gfTokenNum>>24); 
											//only bits 0..7 are compiled!
					compileTCode(gfTokenNum>>16);
					compileTCode(gfTokenNum>>8);
					compileTCode(gfTokenNum);
				}
			}
			gfTokenStart = gfTibPos;
			continue;
		}
// could not parse as number, try to interpret as token
		else if (findUserToken()) {	//User Token found, index in gfParsed
			if(interpretUserToken())  return;	//returns 0 if ok !!
			gfTokenStart = gfTibPos;
			continue;
		}
		else  {
			if (findKernelToken() == 0) {	// Kernel Tokenstring not found
			gfErrors |= ERR_STRINGTOKEN;
			gfCodeIndex = gfCodeStartIndex;
			gfTokenStart = gfTibPos;
			return;
			}
//Kernel Tokenstring found;
			interpretKernelToken();	//Kernel token found, index in gfParsed
			if (gfErrors) return;
			gfTokenStart = gfTibPos;
			continue;
		}
	}
}


void backProcEval(void)   // **************************************************
//while User Token is executing, evaluate "background process" terminal input
//in case of errors, send a message but don't set an error flag
// ************************************************************************
{
uint32_t k;
const uint8_t invalidStr [] = " <-not a Kernel OP nor a number";
const uint8_t noBackStr [] = " <-invalid for Back-Ctrl";
const uint8_t runErrStr [] = " <-Runtime Error";

	for (k=82;k<gfTibMax;k++) sendTB(gfTib[k]);	//for info, send parsed string

	gfTibPos = 82;
	gfComment = 0;
	gfTokenStart = gfTibPos;

	while (gfTibPos < gfTibMax)  {
		scanNext();
		if (gfTokenstr[0] == 0) continue;	//ignore CTRL_B,TAB,LFF,ESC	

		if (interpretNumber()) {	//valid number parsed, here only interpreting
			gfBackProcData = gfTokenNum;
			if (gfExecUser) gfBackNumValid = 1;
			gfTokenStart = gfTibPos;
			continue;
		}
		else  {
//result == 0: could not parse as number, try to interpret as token
			if (findKernelToken() == 0) {	// Kernel Tokenstring not found
				sendString(invalidStr);
				goto ERROR_BREAK;
			}
			//else Kernel Tokenstring found;
			if (gfKernelAttributes[gfParsed] & A_BAK) {	//only if BAK attribute
				interpretKernelToken();	//Kernel token found, index in gfParsed
				if (gfErrors)  {
					gfErrors = 0;		//don't cancel background process
					sendString(runErrStr);
					goto ERROR_BREAK;
				}
			}
			else  {
				sendString(noBackStr);
				goto ERROR_BREAK;
			}
			gfTokenStart = gfTibPos;
			continue;
		}
	}
ERROR_BREAK:
	sendTB(CRR);
	gfTibMax = 82;
	gfTibPos = 82;
	gfComment = 0;
}


void scanNext(void)   // **************************************************
// updates gfTokenstr[] with the next name string in TIB
// ************************************************************************
{
uint32_t k, tib;

	while (gfTib[gfTokenStart] == SPACE) gfTokenStart++;
								//eliminate leading SPACEs

	if (gfTib[gfTokenStart] == 0)  {	//is the case if CTRL_B,TAB,LFF,ESC
		gfTokenstr[0] = 0;
		gfTibPos = gfTokenStart;
		return;
	}

	k = 0;
	while ((tib = gfTib[gfTokenStart+k]) != SPACE) {
								//search trailing SPACE(s) after scanned token
		gfTokenstr[k] = tib;
		k++;
		if (k > 14) break;		//limit length of string Bugfix 26May2022
						//(max.length of number. Code names are limited later
	}
	gfTokenstr[k] = 0;
	gfTibPos = gfTokenStart + k;
}


uint32_t interpretNumber(void)  //	*****************************************
// hex numbers can be entered without leading 0X, gfTempNumBase is criterium
// optionally, when entered with leading 0X, Hex base is assumed temporarily
// leading '-'(after 0x if present) is accepted(good for small negative numbers)
// to simplify parameter transfer, succesfully parsed number is in "gfTokenNum"
//return:
// 1 = valid number
// 0 = no valid number, try to interpret as command token
// *************************************************************************
{
uint32_t numLen;

	numLen = 0;
	while (gfTokenstr[numLen] != 0) numLen++;

	if ((gfTokenstr[0] == '-') && (numLen == 1))  return 0;	 //"-" operator 

	if ((gfTokenstr[0] == '0') &&
					((gfTokenstr[1] == 'X') || (gfTokenstr[1] == 'x'))) {
		if (gfTokenstr[2] == '-')  {
			//bugfix 11June22:wrong handling length of big negative numbers
			if (numLen > 11)  return 0;
		}
		else if (numLen > 10)  return 0;	//try to parse as command token
		if (string2Int(&gfTokenstr[2], 16))  return 1;
					// success: interpreted as number, result in "gfTokenNum"
	}
	else if (gfTokenstr[0] == '%') {		//new feature 19Jan24
		if (gfTokenstr[1] == '-')  {
			if (numLen > 12)  return 0;
//			if (numLen > 12)  return 2;		//too long, not accepted as name
		}
		else if (numLen > 11)  return 0;	//try to parse as command token
//		  else if (numLen > 11)  return 2;	//too long, not accepted as name
		if (string2Int(&gfTokenstr[1], 10))  return 1;
					// success: interpreted as number, result in "gfTokenNum"
	}
	else {
		if (gfTempNumBase == 16)  {
			if (gfTokenstr[0] == '-')  {
				if (numLen > 9)  return 0;
			}
			else if (numLen > 8)  return 0;	//try to parse as command token
		}
		else  {
			if (gfTokenstr[0] == '-')  {
				if (numLen > 11)  return 0;
			}
			else if (numLen > 10)  return 0;	//name is too long
		}
		if (string2Int(&gfTokenstr[0], gfTempNumBase)) return 1;
					// success: interpreted as number, result in "gfTokenNum"
	}
	return 0; //string doesnt represent a number:try to parse as command token
}


uint32_t string2Int(uint8_t* numStr, uint32_t numBase)  //	******************
//return: 1= valid number, 0=no valid number, try to interpret as command token
// ***************************************************************************
{
uint32_t n, length;
uint8_t numString[16];
uint32_t digit;

	n = 0;
	for (length = 0; length < 16;length++) {
		numString[length] = numStr[length];
	//transform into capital letters:
		if ((numString[length] >= 'a') && (numString[length] <= 'z'))
													numString[length] -= 0x20;
		if (numString[length] == 0)  break;
	}
	numString[15] = 0;				//in any case, doesn't harm if length<15
	if ((numString[0] == '-') && (length == 1))  return 0;	//"-" operator

	gfTokenNum = 0;
	n = 0;
	if (numString[0] == '-') n = 1;
	while (n < length) {
		digit = checkDigit(numString[n], numBase);
		if (digit == 0xFFFFFFFF) {
			return 0; //string doesn't represent a number
		}
		gfTokenNum = (int32_t)((gfTokenNum * numBase) + digit);
		n++;
	}
	if (numString[0] == '-') {
		gfTokenNum = -gfTokenNum;
	}
	return 1;						//string is parsed as valid number
}


uint32_t checkDigit(uint8_t digit, uint32_t numBase)   // *******************
// **************************************************************************
{
	if (digit < 0x30) return 0xFFFFFFFF;
	digit -= 0x30;
	if (digit < 10) return (int32_t)digit;
	if (numBase == 10) return 0xFFFFFFFF;
	else {
		if (digit < 0x11) return 0xFFFFFFFF; //error if digit < 0x41 (='A')
		if (digit > 0x16) return 0xFFFFFFFF; //error if digit > 0x46 (='F')
		digit -= 7;							//range 0....0x0F
	}
	return digit;
}


uint32_t findKernelToken(void)  //	****************************************
// *************************************************************************
{
uint32_t k, n;
uint8_t kernelChar, refChar;

	k = 0;					//index 0 is op_ERR
	while (k < KERNELSIZE) {
		for (n=0; n < 8;n++) {
			kernelChar = gfKernelNames[k][n];
			refChar = gfTokenstr[n];
//transform to capital letters:
			if ((kernelChar >= 'a')&&(kernelChar <= 'z')) kernelChar -= 0x20;
			if ((refChar >= 'a')&&(refChar <= 'z')) refChar -= 0x20;
			if (kernelChar == SPACE) kernelChar = 0;
			if (kernelChar != refChar) break;
			if (refChar != 0) continue;
			gfParsed = k;			 //="gfKernelNames[]"index of found token
					//=same index in "gfKernelOps[]" and "gfKernelAttibutes[]"
			return 1;
		}
		k++;
	}
	return 0;
}


uint32_t findUserToken(void)   // ******************************************
// *************************************************************************
{
int32_t k;
uint32_t n;
uint8_t refChar, userChar;

	k = 0;

	while (gfUserHeader[k].name[0] != 0) k++;
	k--;							//k is index of last compiled USER Token

	while (k != -1)  {				//start search from the last compiled
		for (n=0; n < 8;n++) {
			userChar = gfUserHeader[k].name[n];
			refChar = gfTokenstr[n];
//transform into capital letters:
			if ((refChar >= 'a')&&(refChar <= 'z')) refChar -= 0x20;
			if (userChar != refChar)  break;
			if (userChar == 0) {
				gfParsed = k;		//="gfUserHeader[]" index of found token
				return 1;
			}
		}
		k--;
	}
	return 0;
}


void interpretKernelToken(void) // *****************************************
// *************************************************************************
{
	if (gfKernelAttributes[gfParsed] & A_RUN) {
		gfErrors |= ERR_RUNTOKEN;
		return;
	}

	if(gfCompiState == 0) {	 //interpreting, executing threaded code
		if (gfKernelAttributes[gfParsed] & A_CPO) {
			gfErrors |= ERR_COMPONLY;
			return;
		}
		gfKernelOps[gfParsed]();		//execute normally
	}
	else {				// compiling is active
		if (gfKernelAttributes[gfParsed] & A_IME) {
			gfKernelOps[gfParsed]();	//don't compile, execute immediately
		}
		else  compileTCode(gfParsed);	//compile this token
	}
}


uint32_t interpretUserToken(void) // ***************************************
//essential global parameter:gfParsed 
// unusual return value: 0= ok, else=Error
// *************************************************************************
{
uint32_t oldTibMax, oldTibPos;

	if(gfCompiState == 0) {	//interpreting, COMPONLY Option not accepted
		if ((gfUserHeader[gfParsed].attributes & A_VARCON) == 0) {
			gfIP = (uint8_t*)gfUserHeader[gfParsed].toCompile;
			gfExecUser = 1;			//modify query() behaviour
			oldTibMax = gfTibMax;
			oldTibPos = gfTibPos;
			gfTibMax = 82;			//intermediate TIB for background process
			gfTibPos = 82;
			gfComment = 0;

			executeTCode();
			gfTibMax = oldTibMax;	
			gfTibPos = oldTibPos;
			return gfErrors;							//=0 if everything OK
		}
		else  {											//VARCON
			gfSP++;
			*gfSP = gfUserHeader[gfParsed].toCompile;	//special;
			return 0;									// everything OK
		}
	}
	else {												// compiling
			compileTCode(gfParsed | 0x80);		
								//18Jan25check for A_VARCON in executeTCode()
			return gfErrors;							//=0 if everything OK
	}
	return gfErrors;						//this exit should be impossible;
}


uint32_t compileUserSymbol(uint32_t attribute)  // *************************
// **************************************************************************
{
uint32_t n;
uint8_t userString[8];
const uint8_t warnStr [] = " Warn: will override Kernel OP w.same name";

	if (gfNextNameCompi == MAXUSER)  {
		gfErrors |= ERR_NUMVAL;
		return 0;
	}
	gfTokenStart = gfTibPos;
	scanNext();
	gfTokenStart = gfTibPos;

   if (interpretNumber() != 0) {
		gfErrors |= ERR_CONTEXT;
		return 0;
	}

	if (findKernelToken()) {
		sendString(warnStr);
		if (conFirm() == 0)  return 2;	  //not compiled, no error flag set
	}

	for (n = 0;	n < 8;n++) {
		userString[n] = gfTokenstr[n];
//transform into capital letters:
		if ((userString[n] >='a')&&(userString[n] <='z'))userString[n] -=0x20;
		if (userString[n] == 0) break;
		if (n == 7)  {
			gfErrors |= ERR_SYNTAX;	//name string too long
			return 0;
		}
	}
	n = 0;
	while ((userString[n] != 0) && (n < 7))  {
	  gfUserHeader[gfNextNameCompi].name[n] = userString[n];
	  n++;
	}
	gfUserHeader[gfNextNameCompi].name[n+1] = 0;
	gfUserHeader[gfNextNameCompi].toCompile=(uint32_t)(&gfTCode[gfCodeIndex]);
	gfUserHeader[gfNextNameCompi].attributes = gfCodeIndex | attribute;
	return 1;
}


void compileTCode(uint32_t token) // *************************************
// *************************************************************************
{
	gfTCode[gfCodeIndex] = (uint8_t)(token & 0x000000FF);
	gfCodeIndex ++;
	if (gfCodeIndex >= CODETOP)  gfErrors |= ERR_USERCODE;
}


void executeTCode(void)   // ***********************************************
// *************************************************************************
{
uint32_t tibstate;
uint32_t index;

const uint8_t finiStr [] = " debug finished ";

	while (gfIP < gfTCode + USERTOP)  {
		if (gfBackProcess == 0)  {
			if (*gfIP < KERNELSIZE) gfKernelOps[*gfIP]();	//is Kernel Token
			else {								//then it is a User Token
				index = *gfIP & 0x0000007F;		//new 18Jan25:
				if (gfUserHeader[index].attributes & A_VARCON) {	
					gfSP++;
					*gfSP = gfUserHeader[index].toCompile;
				}
				else  intoUserCode();
				}
		}
		else  {
			if (checkTerminal())  {
				tibstate = subQuery();
				if (tibstate == 1) {			//Terminal CRR input
					backProcEval();
				}
				if (tibstate == 2) {			//Terminal ESC input
					gfErrors |= ERR_CANCEL;
					return;
				}
//tibstate = 3(TAB=step) ignored in this case
			}
			if (*gfIP < KERNELSIZE)  {			//is Kernel Token
				if (gfBackProcess == 3)  execNext();
				else  gfKernelOps[*gfIP]();
			}
			else {								//then it is a User Token
				index = *gfIP & 0x0000007F;		//new 18Jan25:
				if (checkVarcon(index)) intoUserCode();
			}
		}

//serve Background process
		if (gfStepSpeed)  {
			if ((gfUserHeader[4].toCompile > 0) ||
								(gfUserHeader[4].toCompile == 0xFFFFFFFE)) {
				if (gfSTEPCOUNT == 0) subMOT();
			}
		}
		if (gfErrors) return;
		if (gfIP)  {			//modified 29Jan25
			gfIP ++;
			continue;
		}
		else {					//execution finished by SEMIS or RETURN
			if (gfBackProcess == 3) sendString(finiStr);
			if (gfBackProcess) gfBackProcess = 1;	//bugfix 26Jan25
			gfExecUser = 0;
			gfBackNumValid = 0;
			gfNestLevel = 0;
			return;
		}
	}
}


void intoUserCode(void)   // ***********************************************
// *************************************************************************
{
uint32_t index;

	gfNestLevel++;
	gfRP ++;
	*gfRP = (uint32_t)gfIP;						//gets restored by "op_SEMIS"
	index = *gfIP & 0x0000007F;
	gfIP = (uint8_t*)gfUserHeader[index].toCompile;

	if (*gfIP < KERNELSIZE)  {					//is Kernel Token  
		if (gfBackProcess != 3)  gfKernelOps[*gfIP]();
		else  execNext();
	}
	else  {										//then it is a User Token
		index = *gfIP & 0x0000007F;				//new index in next UserToken!
		if (checkVarcon(index)) intoUserCode();	//new 18Jan25

	}
}


uint32_t checkVarcon(uint32_t index)   // **********************************
//new 18Jan25
// *************************************************************************
{
uint32_t result = 0;

	if (gfUserHeader[index].attributes & A_VARCON) {	
		gfSP++;
		*gfSP = gfUserHeader[index].toCompile;
		result =  0;
	}
	else  result = 1;
	if (gfBackProcess == 3)  {
		showPreStep(index+1);	//safety against index=0
		if (gfErrors) return result;
		showPostStep();
	}
	return result;
}


void execNext(void)   // ***************************************************
//abbreviation to save runtime if not in "step mode"
// *************************************************************************
{
	showPreStep(0);
	if (gfErrors) return;		//usually ESC = terminate User Thread
	gfKernelOps[*gfIP]();
	showPostStep();
}


void showPreStep(uint32_t index)  // ***************************************
// First the name of the next code token is sent to terminal.
// Execution of this token is triggered by TAB key on terminal software.
// Finally (by "showPostStep") the result is sent to terminal (output + stack)
// can be used as a simple Forth-Decompiler, too
// to speed display up, keep TAB key pressed
// *************************************************************************
{
uint32_t k, ipVal, indent, ipNext, data;
uint32_t *localIP;
const uint8_t IPStr [] = "(IP->";

	if (gfErrors)  return;

	gfIPold = gfIP;							//keep in mind for showPostStep()
	ipVal = *gfIP;
	indent = gfNestLevel * 3;

	sendTB(CRR);
	for (k=0; k < indent; k++) sendTB(SPACE);
	sendNX32((uint32_t)gfIPold & 0x0000FFFF);
	sendTB(SPACE);

	if (index) {					//new 18Jan25
		index -= 1;				//was incremented to fit for userHeader[0]
		index2String(index);
		sendTB(0x28);	 //'('
		sendTB('=');
		if (gfTempNumBase==16) send0X32(gfUserHeader[index].toCompile);
		else  sendNS32(gfUserHeader[index].toCompile);
		sendTB(0x29);	 //')'
	}
	else  {
		sendNX32 (ipVal & 0x0000FFFF);
		sendTB(SPACE);
		if (ipVal > KERNELSIZE)  {			//user token
			sendTB('-');
			sendTB('>');
			sendTB(SPACE);
		}
		token2String(ipVal);
	}

	if (((ipVal == IDX_DOIF) && (*gfSP == 0)) ||
			((ipVal == IDX_DOCASE) && (*gfSP != *(gfSP-1))) ||
			((ipVal == IDX_DOLTCAS) && (*(gfSP-1) >= *gfSP)) ||
			((ipVal == IDX_DOGTCAS) && (*(gfSP-1) <= *gfSP)) ||
			(ipVal == IDX_DOELSE) ||
			((ipVal == IDX_DOUNTIL) && (*gfSP == 0)) ||
			((ipVal == IDX_DOWHILE) && *gfSP) ||
			(ipVal == IDX_DOAGAIN) || 
			(ipVal == IDX_DOLOOP)) {
		sendString(IPStr);
		sendNX32(((*(gfIP+1)<< 8) |	*(gfIP+2)) + 1);
						//gfIP is incremented before "executeTCode()" is left
		sendTB(0x29);						//')'
	}

	if (ipVal==IDX_DOLIT8B) {
		ipNext = (uint32_t)(*(gfIP+1));
		sendTB(0x28);						//'('
		if (gfTempNumBase==16)  send0X32(ipNext);
		else  sendNS32(ipNext);
		sendTB(0x29);						//')'
	}

	if (ipVal==IDX_DOLIT) {
		ipNext = (uint32_t)(*(gfIP+1) << 24);
		ipNext |= (uint32_t)(*(gfIP+2) << 16);
		ipNext |= (uint32_t)(*(gfIP+3) << 8);
		ipNext |= (uint32_t)*(gfIP+4);
		sendTB(0x28);						//'('
		if (gfTempNumBase==16)  send0X32(ipNext);
		else  sendNS32(ipNext);
		sendTB(0x29);						//')'
	}

	if (ipVal==IDX_DOWVA) {
		sendTB(0x28);						//'('
		ipNext = (uint32_t)(*(gfIP + 1) & 0x7F);
		index2String(ipNext);
		sendTB('@');
		data = (uint32_t)&gfUserHeader[ipNext].toCompile;
		send0X32(data & 0x0000FFFF);
		sendTB(0x29);	 //')'
	}

	if (ipVal==IDX_SEMIS) {
		sendTB(SPACE);
		sendTB('<');
		sendTB('-');
	}

	k = 0;
	while (k != 3) {			//while terminal input != TAB
		if (gfBackProcess < 3) break;
		servSystem();
		if (checkTerminal())  {
			k = subQuery();
			if (k == 1)   {		//Terminal CRR input
				backProcEval();
				continue;
								//background operation does not release step
			}
			if (k == 2)  {		//Terminal ESC input
				gfErrors |= ERR_CANCEL;
				return;
			}
//new 31Dec23:
			if (k == 4)  {		//Terminal CTRL_R input
				if (gfNestLevel != 0)  {
					gfBackNestLevel = gfNestLevel;
					gfBackProcess = 2;
				}
				else  k = 3; //continue;
			}
		}
	}

	sendTB(SPACE);
	if ((ipVal >= IDX_CONTI) && (ipVal <= IDX_ZBREAK)) {
		sendString(IPStr);
	}
	else  {
		sendTB('{');
		sendTB(SPACE);
	}
	return;
}


void showPostStep(void)  //	************************************************
// remark see "showPreStep()"
// *************************************************************************
{
uint32_t n, ipVal;

	if (gfErrors)  return;

	ipVal = *gfIPold;

	if ((ipVal >= IDX_CONTI) && (ipVal <= IDX_ZBREAK)) {
		sendNX32(((uint32_t)(gfIP+1)) & 0x00000FFF);
						//gfIP is incremented before "executeTCode()" is left
		sendTB(0x29);	
		return;
	}
	sendTB('}');
	sendTB(SPACE);
	op_PRINTSTACK();

	if ((ipVal >= IDX_DODO) && (ipVal <= IDX_RP4DEC))  {
		sendTB(SPACE);
		sendTB('R');
		sendTB(':');
	}
	if ((ipVal == IDX_DOFOR) ||	(ipVal == IDX_DOLOOP) ){
		if (((uint32_t)(gfRP - gfReturnStack) > 2)) {
			sendNX32((*(gfRP-3) & 0x0000FFFF));
			sendTB(SPACE);
			sendTB(SPACE);
			sendNX32((*(gfRP-2) & 0x0000FFFF));
			sendTB(SPACE);
			sendTB('i');
			sendNX32((*(gfRP-1) & 0x0000FFFF));
			sendTB(SPACE);
			sendTB('x');
			sendNX32((*gfRP & 0x0000FFFF) + 1);
			//the compiled jump	address is 1 less than effectively executed
		}
	}
	else if ((ipVal >= IDX_DODO) && (ipVal <= IDX_DOAGAIN))
											sendNX32(*gfRP & 0x0000FFFF);

}


void emptyForth(void)//	****************************************************
//load and install an empty project with default settings
//Two CONSTants P0, P1 and two VARs V2, V3 are installed as user token
// *************************************************************************
{
uint32_t k;
uint32_t *pointer;

	gProject = 0;
	gfBackProcess = 0;				//default mode changed 26Jan25
	gfAuto = 0;	

//default SPI setup:
	gfSpiMode = 0;
	gfSpiBits = 8;
	gfSpiClock = 0;		//default = MAX = about 40kHz clock (no delay loops)

//default PWM setup:
	gfFreq = 0;			//default = max = 9000 Hz (CLOCK_48M:24000 Hz/15000Hz)
	gfPWM1 = -1;		//-1 is OFF, PWM max 100 (CLOCK_48M:2000)
	gfPWM2 = -1;		//-1 is OFF
	gfStepSpeed = 0;

//compile 4 VARCONs for general use
	for (k = 0; k < 4;k++) {
		gfUserHeader[k].name[0] = 'V';
		gfUserHeader[k].name[1] = k + 0x31;
		gfUserHeader[k].name[2] = 0;
		gfUserHeader[k].attributes = A_VARCON;
		gfUserHeader[k].toCompile = 0;
	}	
//compile VARCON NSTP(number of steps) for op_MOT background control
	gfUserHeader[4].name[0] = 'N';
	gfUserHeader[4].name[1] = 'S';			
	gfUserHeader[4].name[2] = 'T';			
	gfUserHeader[4].name[3] = 'P';
	gfUserHeader[4].name[4] = 0;
	gfUserHeader[4].attributes = A_VARCON;
	gfUserHeader[4].toCompile = -1;		//start with DC	motor
	gfNextNameCompi = 5;
	gfUserHeader[5].name[0] = 0;
	gfCodeIndex = 0;
	op_ABORT();
}


// arithmetic ops  #########################################################

void op_ERR(void)  // *****************************************************
// ( -- )
// does nothing, except consume a little bit of runtime
// *************************************************************************
{
	gfErrors |= ERR_USERCODE;
}

void op_DROP(void) // *****************************************************
// ( x1 x2 -- x1)
// forget the TOS and pop all lower stack entries one up
// *************************************************************************
{
	gfSP--;
	if (gfSP < gfDataStack + 3) gfErrors |= ERR_STACK;
}


void op_DUP(void)  // ******************************************************
// ( x -- x x)
// push all stack entries one down and copy the new NOS as TOS
// *************************************************************************
{
	gfSP++;
	if (gfSP >= gfDataStack+MAXDATASTACK - 4)  {
		gfErrors |= ERR_STACK;
		return;
	}
	*gfSP = *(gfSP-1);
}


void op_SWAP(void) // *****************************************************
// (x1 x2 -- x2 x1)
// exchange TOS and NOS
// *************************************************************************
{
uint32_t TEMP;

	TEMP = *(gfSP-1);
	*(gfSP-1) = *gfSP;
	*gfSP = TEMP;
}


void op_OVER(void) // *****************************************************
// ( x1 x2 -- x1 x2 x1)
// push all stack entries one down and copy the new 3rd of stack to TOS
// *************************************************************************
{
	gfSP++;
	*gfSP = *(gfSP-2);
}


void op_ROT(void)  // ******************************************************
// ( x1 x2 x3 -- x2 x3 x1 )
// rotate 3rd stack entry up
// *************************************************************************
{
uint32_t TEMP;

	TEMP = *(gfSP-2);
	*(gfSP-2) = *(gfSP-1);
	*(gfSP-1) = *gfSP;
	*gfSP = TEMP;
}


void op_PICK(void) // *****************************************************
// (  xn x(n-1) x2 x1 n  ---  xn x(n-1) ... x2 x1 x0 xn  )
// keep	 n(=TOS) in mind and copy the (n+1)th element of stack on TOS
// 0 PICK performs DROP
// 1 PICK effectively performs DUP
// 2 PICK effectively performs OVER
// *************************************************************************
{
int32_t temp;

//first check available depth of stack
	temp = (int32_t)(gfSP - gfDataStack - 4) - (int32_t)*gfSP;
	if (temp < 0) {
		gfErrors |= ERR_STACK;
		return;
	}
	temp = *gfSP;
	if (temp == 0) {
		gfSP--;
		return;
	}
	*gfSP = *(gfSP - temp);
}


void op_ROLL(void) // *****************************************************
// (  xn x(n-1)...x2 x1 n --- x(n-1)  x2 x1 xn )
// drops TOS, but keeps n in mind. Rotates the (new) n-th stack element 
// up to TOS. 
// 0 ROLL and 1 ROLL do nothing except dropping TOS
// 2 ROLL is equivalent to SWAP
// 3 ROLL is equivalent to ROT
// bigger values of "n" provide manipulation of deep stack items
// e.g. the sequence  "4 ROLL 4 ROLL" effectively performs 2SWAP
// *************************************************************************
{
uint32_t k, temp;

//first check available depth of stack
	k = (int32_t)(gfSP - gfDataStack - 4) - (int32_t)*gfSP;
	if (k >= 0x80000000) {
		gfErrors |= ERR_STACK;
		return;
	}

	k = *gfSP;
	gfSP--;
	if (k < 2) return;
	temp = *(gfSP - k + 1);	
	while(k)  {
		*(gfSP - k + 1 ) = *(gfSP - k + 2);
		k -- ;
	}
	*gfSP = temp;
}


void op_INJECT(void)  // ****************************************************
// (  x(n+1) xn... x2 x1 n ---  x1 x(n+1) xn... x2 )
// drops TOS, keeps n in mind. 
// then rotates (new) TOS(= x1) down below the (actual) n-th stack element.
// 0 INJECT and 1 INJECT simply drop TOS
// 2 INJECT effectively performs SWAP
// 3 INJECT effectively performs -ROT
// bigger values of "n" provide manipulation of deep stack items
// e.g.sequence "n PICK n INJECT" effectively doubles the deep n-th stack item
// **************************************************************************
{
uint32_t k, n, temp;

//first check available depth of stack
	k = (int32_t)(gfSP - gfDataStack - 4) - (int32_t)*gfSP;
	if (k >= 0x80000000) {
		gfErrors |= ERR_STACK;
		return;
	}

	temp = *(gfSP - 1);
	n = *gfSP;
	gfSP --;
	if (n < 2) return;
	for (k=0;k < (n-1);k++)  {
		*(gfSP - k) = *(gfSP - k - 1);
	}
	*(gfSP - n + 1) = temp;
}


void op_PLUS(void) // *****************************************************
// ( x1 x2 -- x1+x2)
// *************************************************************************
{
	*(gfSP-1) = (int32_t)*(gfSP-1) + (int32_t)*gfSP;
	gfSP--;
}


void op_INCT(void) // *****************************************************
// ( x -- x+1)
// *************************************************************************
{
	*gfSP = *gfSP + 1;
}


void op_MINUS(void)  //	****************************************************
// ( x1 x2 -- x1-x2)
// *************************************************************************
{
	*(gfSP - 1) = (int32_t)*(gfSP - 1) - (int32_t)*gfSP;
	gfSP--;
}


void op_DECT(void) // *****************************************************
// ( x -- x-1)
// *************************************************************************
{
	*gfSP = *gfSP - 1;
}


void op_ABS(void)  // ******************************************************
// ( x -- abs(x))
// *************************************************************************
{
int32_t abs;

	abs = (int32_t)*gfSP;
	if (abs < 0) *gfSP = -abs;
}


void op_SHIFT(void)  //	***************************************************
// if (X2>=0) ( x1 x2 -- x1<<x2 )
// if (X2<0) ( x1 x2 -- x1>>x2 )
// abs(x2) <= 31
// *************************************************************************
{
int32_t k,abs;

	k = (int32_t)*gfSP;
	abs = k;
	if (abs < 0) abs = -abs;
		if (abs > 31)  {
			gfErrors |= ERR_NUMVAL;
			return;
		}
	gfSP--;
	if (k >= 0)   {
		*(gfSP) = *(gfSP)<<k;
	}
	else  {
		*(gfSP) = *(gfSP)>>abs;
	}

}


void op_AND(void)  // ******************************************************
// ( x1 x2 -- x1&x2)
// *************************************************************************
{
	*(gfSP-1) &= *gfSP;
	gfSP--;
}

void op_OR(void)  // ******************************************************
// ( x1 x2 -- x1|x2)
// *************************************************************************
{
	*(gfSP-1) |= *gfSP;
	gfSP--;
}


void op_XOR(void)  // ******************************************************
// ( x1 x2 -- x1 XOR x2)
// *************************************************************************
{
	*(gfSP-1) = *gfSP ^	*(gfSP-1);
	gfSP--;
}


void op_NOT(void)  // *****************************************************
// ( x --- <one's complement of x>)
// logical inversion bit by bit (one's complement)
// in contrast to other Forth,this is an arithmetic	operator differing from "0="
// *************************************************************************
{
	*gfSP = ~ (*gfSP);
}


void op_NEG(void)  // *****************************************************
// ( x --- <two's complement of x>)
// mathematical negation (two's complement)
// *************************************************************************
{
	*gfSP = -(int32_t)(*gfSP);
}


void op_MULT(void) // *****************************************************
// ( factor factor -- TOS=factor*factor)
// multiplication is made 32bit signed-i.e.	may overflow with factors>0xB000
// if this case: no error msg, no warning. User is responsible
// *************************************************************************
{
	*(gfSP-1) = (int32_t)*gfSP * (int32_t)(*(gfSP-1));
	gfSP--;
}


void op_DIV(void)  // ******************************************************
// ( dividend divisor  -- x1/x2)
// 32 bit signed division
// *************************************************************************
{
	if (*gfSP == 0)  {
		gfErrors |= ERR_DIVZERO;
		return;
	}
	else  {
		*(gfSP-1) = (int32_t)*(gfSP-1) / (int32_t)*gfSP;
		gfSP--;
	}
}


void op_MULTDIV(void)  // **************************************************
// ( factor factor divisor-- TOS=(factor*factor)/divisor)
// multiplication is made 32bit int: i.e. may overflow with factors>0xB000
// if this case: no error msg, no warning. User is responsible
// *************************************************************************
{
int32_t factor, dividend, divisor;

	if (*gfSP == 0)  {
		gfErrors |= ERR_DIVZERO;
		return;
	}
	else  {
		factor = (int32_t)*(gfSP - 2);
		dividend = (int32_t)*(gfSP-1);
		divisor =  (int32_t)*gfSP;
		dividend = factor *	dividend;
		gfSP -= 2;
		* gfSP = dividend /	divisor;
	}
}


void op_MOD(void)  // ******************************************************
// return the remainder of a 32 bit signed integer division
// ( dividend divisor  -- remainder )
// the remainder has always the same sign as the dividend
// *************************************************************************
{
	if (*gfSP == 0)  {
		gfErrors |= ERR_DIVZERO;
		return;
	}
	else  {
		*(gfSP-1) = (int32_t)*(gfSP-1) % (int32_t)(*gfSP);
		gfSP--;
	}
}


void op_DIVMOD(void)  // ******************************************************
// return the remainder(NOS) and quotient(TOS) of a 32 bit signed int. division
// ( dividend divisor  -- remainder quotient )
// the remainder has always the same sign as the dividend
// *************************************************************************
{
int32_t dividend, divisor;

	if (*gfSP == 0)  {
		gfErrors |= ERR_DIVZERO;
		return;
	}
	else  {
		dividend = (int32_t)*(gfSP-1);
		divisor =  (int32_t)*gfSP;
		*(gfSP-1) = dividend % divisor;
		*gfSP = dividend / divisor;
	}
}


void op_BSET(void) // ****************************************************
// (oldTOS bit# -- newTOS) read modify write
// Top of Stack (TOS) manipulation
// bit# = bit number to be set (0 to 31)
// *************************************************************************
{
	if (*gfSP > 31) {
		gfErrors |= ERR_NUMVAL;
		return;
	}
	*(gfSP-1) |= 1 << *gfSP;		//bitNo-->bitMask and set NOS
	gfSP--;							//NOS-->TOS
}


void op_BCLR(void) // *****************************************************
// (oldTOS bit# -- newTOS) read modify write
// Top of Stack (TOS) manipulation
// bit# = bit number to be set (0 to 31)
// *************************************************************************
{
	if (*gfSP > 31) {
		gfErrors |= ERR_NUMVAL;
		return;
	}
	*(gfSP-1) &= ~(1 << *gfSP);	  //bitNo-->bitMask and invert NOS bitwise
	gfSP --;						//NOS-->TOS
}


void op_BTST(void) // *****************************************************
// (oldTOS bit# -- oldTOS bitvalue)
// Top of Stack (TOS) manipulation
// bit# = bit number to be checked (0 to 31)
// new TOS = (bitvalue = boolean value of this bit (0 or 1)
// *************************************************************************
{
uint32_t temp;

	if (*gfSP > 31) {
		gfErrors |= ERR_NUMVAL;
		return;
	}

	temp = 1 << *gfSP;		//bitNo-->bitMask
	temp = *(gfSP-1) & temp;	
	if (temp) *gfSP = 1;
	else *gfSP = 0;
}


void op_RANDOM(void)  // ****************************************************
//	( x1 x2  -- TRUE|FALSE)
//	TRUE if x1==X2
// *************************************************************************
{
	gfSP ++;
	*gfSP = gfRandom;
}


//comparison ops ###########################################################

void op_EQUAL(void)  //	****************************************************
//	( x1 x2  -- TRUE|FALSE)
//	TRUE if x1==X2
// *************************************************************************
{
	if (*gfSP == *(gfSP-1)) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_UNEQU(void)  //	****************************************************
//	( x1 x2  -- TRUE|FALSE)
//	TRUE if x1!=X2
// *************************************************************************
{
	if (*gfSP != *(gfSP-1)) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_ZEQ(void)  // ******************************************************
//	( x   -- TRUE|FALSE)
//	TRUE if x==0, FALSE	if x!=0
// *************************************************************************
{
	if (*gfSP == 0) *gfSP = 1;
	else *gfSP = 0;
}


void op_GRTR(void) // *****************************************************
//	( x1 x2  -- TRUE|FALSE)
//	signed compare
//	TRUE if x1>X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) > (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_GRTREQ(void)  // ***************************************************
//	( x1 x2  -- TRUE|FALSE)
//	signed compare
//	TRUE if x1>=X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) >= (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_LESS(void) // *****************************************************
//	( x1 x2  -- TRUE|FALSE)
//	signed compare
//	TRUE if x1<X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) < (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_ULESS(void)  //	*****************************************************
//	( x1 x2  -- TRUE|FALSE)
//	unsigned compare
//	TRUE if x1<X2
// *************************************************************************
{
	if ((uint32_t)*(gfSP-1) < (uint32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_LESSEQ(void)  // ***************************************************
//	( x1 x2  -- TRUE|FALSE)
//	signed compare
//	TRUE if x1<=X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) <= (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


// memory ops  #############################################################

void op_VARCON(void)  // ****************************************************
// ( value -- )
// syntax example:
// 23 VARCON CONNY
// defines the VariableConstant CONNY and initializes it with value 23
// at runtime, a VARCON puts its value on TOS
// *************************************************************************
{
	if (compileUserSymbol(A_VARCON) == 0)  {
		gfErrors |= ERR_CONTEXT;
		return;
	}
	gfUserHeader[gfNextNameCompi].toCompile = *gfSP;
	gfSP--;
	gfNextNameCompi++;
	return;
}


void op_WVA(void)  // **************************************************
// set a new value to a VARCON
// special non-Forth typical syntax,
// example:
// 23 VARCON TEST
// 0xABCDE W TEST		//""W" is user level operator which triggers op_WVA
// (TEST must be scanned by W to get its .toCompile address)
// TEST .S  =returns 0xABCDE on stack
// the new value is inserted at any place where VARCON is threaded
// and will be used in future until a new value is inserted
// *************************************************************************
{
uint32_t addr;

	gfTokenStart = gfTibPos;
	scanNext();
	if (findUserToken() == 0) {		//updates "gfParsed"=index of gfUserHeader
		gfErrors |= ERR_USERCONTEXT;
		return;
	}
	gfTokenStart = gfTibPos;

	if ((gfUserHeader[gfParsed].attributes & A_VARCON) == 0)  {
		gfErrors |= ERR_CONTEXT;
		return;
	}

	if (gfExecUser) checkBackNum();

	if (gfCompiState == 0) {		//interpreting
		gfUserHeader[gfParsed].toCompile = *gfSP; //re-init with stack value
		gfSP--;
	}
	else  {		//compiles runtime "W" behaviour
		compileTCode(IDX_DOWVA);
		compileTCode(gfParsed);	
					//"gfUserHeader[gfParsed].toCompile" referenced by index !
					//different from G4xx version to save code length !!
					//compileTCode() compiles only 8bit token here !
	}
}

// terminal ops  ###########################################################

void op_KEYQ(void) // ****************************************************
//check if a character is received via serial terminal port (USART)
//if NOT return 0 on TOS
//if YES return(number of bytes in RX buffer) on TOS
//NO data byte is extracted from terminal input buffer
// *************************************************************************
{
	gfSP++;
	if (gfBackProcess && gfExecUser)  *gfSP = gfTermCheck;		
					  //exclusively set '1'	in subQuery(), cleared in op_KEY()
	else *gfSP = checkTerminal();
}


uint32_t checkTerminal(void)  // *******************************************
//check if a character is received via serial terminal port (USART)
//if NOT return 0
//if YES return number of bytes in RX buffer
// *************************************************************************
{
	if (gUSART_RXTAIL==gUSART_RXHEAD) return 0;
	else  {
		if (gUSART_RXTAIL < gUSART_RXHEAD) {
			return ((uint32_t)(gUSART_RXHEAD - gUSART_RXTAIL));
		}
		else return (USART_RXBUFLEN+(uint32_t)(gUSART_RXTAIL-gUSART_RXHEAD));
	}
}


void op_KEY(void)  // ******************************************************
//wait until a character is received via serial terminal port (USART)
//and return it on the stack
// *************************************************************************
{
uint32_t recvword;

	if (gfBackProcess && gfExecUser)  {
		if (gfTermCheck)  {
			gfSP++;
			*gfSP = gfTermChar;		//set in subQuery()
			gfTermCheck = 0;
		}
	}
	else  {
		while ((recvword = pullTermRxBuf())==0); 
		gfSP++;
		*gfSP = recvword & 0x000000FF;
	}
}


void op_EMIT(void) // *****************************************************
//send the lower 8 bits of TOS as BYTE!! via serial terminal port
// *************************************************************************
{
	sendTB((uint8_t)(*gfSP & 0x000000FF));
	gfSP--;
}


void op_PRINT(void)  //	****************************************************
//send the TOS number as ASCII text via serial terminal port ( USART)
//use system number base
//numbers in the range 0x20000000 to 0x50060C00 always are printed hex formatted
// *************************************************************************
{
	if ((gfTempNumBase == 16)||((*gfSP >= 0x20000000)&&(*gfSP < 0x50060C00))) {
		 send0X32(*gfSP);
	}
	else sendNS32(*gfSP);
	sendTB(SPACE);
	gfSP--;
}


void op_PRINTHEX(void) // *************************************************
//print the TOS number as hex style ASCII text via via serial terminal port
// *************************************************************************
{
	send0X32(*gfSP);
	sendTB(SPACE);
	gfSP--;
}


void op_PRINTDEC(void) // *************************************************
//print the TOS number as hex style ASCII text via via serial terminal port
// *************************************************************************
{
	sendS32(*gfSP);;
	gfSP--;
}



void op_PRINTSTR(void)  //	*********************************************
//compile the subsequent string to be sent at runtime via via serial terminal
//Note: between P" and the first character of the string MUST be a SPACE
//Non-printable characters (plus " and \) are entered as \+HexCode (w.o. 0x)
//essential examples \5C sends a Backslash, \20  sends a SPACE, \22 sends '"'
// *************************************************************************
{
uint32_t k, n, hibyte, lobyte, word;

	gfTokenStart = gfTibPos + 1;
	scanString();

	k= 0;
	n = 0;
	while (1)  {
		if (gfTokenstr[k] == 0x5C) n++;		//Backslash '\'
						//each Backslash "transports" 3 bytes in Tokenstr[]
		if (gfTokenstr[k] == '"')  {
			gfTokenstr[k] = 0;
			break;
		}
		k++;
		if (k >= 80) {
			gfErrors |= ERR_SYNTAX;
			return;
		}
	}
	if (gfCompiState) {							//compiling
		compileTCode(IDX_DOSTR);
		gfTCode[gfCodeIndex]= k - (n * 2);	//string length count
		gfCodeIndex ++;
		k = 0;
		do {
			if (gfTokenstr[k] != 0x5C) {	//not Backslash '\'
				compileTCode(gfTokenstr[k]);
				k++;
			}
			else  {
				lobyte = packBackslash(k);
				if (lobyte == 0xFFFFFFFF)  {
					gfErrors |= ERR_SYNTAX;
					return;
				}
				k += 3;
				compileTCode(lobyte);
			}
		} while (gfTokenstr[k] != 0);
	}
	else  {									//interpreting
		k = 0;
		do {
			if (gfTokenstr[k] != 0x5C) {	   //'\'
				sendTB(gfTokenstr[k]);
				k++;
			}
			else  {
				lobyte = packBackslash(k);
				if (lobyte == 0xFFFFFFFF)  {
					gfErrors |= ERR_SYNTAX;
					return;
				}
				k += 3;
				sendTB(lobyte);
			}
		} while (gfTokenstr[k] != 0);
	}
}


void scanString(void) // ********************************************
// ************************************************************************
{
uint32_t k;

	k=0;
	while (gfTib[gfTokenStart+k] != 0x22)  {	
								//search trailing '"' after scanned token
		gfTokenstr[k] = gfTib[gfTokenStart+k];
		k++;
		if (k > 78) break;		//limit length of string
	}
	gfTokenstr[k] = 0x22;		//Apostroph
	k++;
	gfTokenstr[k] = 0;
	gfTibPos = gfTokenStart + k;
	return;
}


uint32_t packBackslash(uint32_t k)  //	***********************************
// ************************************************************************
{
uint32_t hibyte, lobyte;
	k++;
	hibyte = gfTokenstr[k];
	if ((hibyte >= 'a') && (hibyte <= 'z')) hibyte -= 0x20;
	if (hibyte < 0x30) return 0xFFFFFFFF;
	hibyte -= 0x30;
	if (hibyte > 9) hibyte -= 7;
	if (hibyte > 0x0F) return 0xFFFFFFFF;
	k++;
	lobyte = gfTokenstr[k];
//	k++;
	if ((lobyte >= 'a') && (lobyte <= 'z')) lobyte -= 0x20;
	if (lobyte < 0x30) return 0xFFFFFFFF;
	lobyte -= 0x30;
	if (lobyte > 9) lobyte -= 7;
	if (lobyte > 0x0F) return 0xFFFFFFFF;
	lobyte += hibyte * 16;
	return lobyte;
}


void op_DECIM(void)  //	****************************************************
//set the system number base to DECIMAL
//Forth internal operations use gTempNumBase, which is usually=gNumBase
//if op_Decim is called during compilation, only gTempNumBase is changed
//op_Semis() or the error handler restores gTempNumBase to gNumBase 
//during compilation number base may temporarily change without global change
// *************************************************************************
{
	gfTempNumBase = 10;
	if (gfCompiState== 0) gfNumBase = 10;
}


void op_HEX(void)  // ***********************************************
//set the system number base to HEX
//comment see op_DECIM
// *************************************************************************
{
	gfTempNumBase = 16;
	if (gfCompiState== 0) gfNumBase = 16;
}


void op_TXON(void)  // ***********************************************
//re-enable terminal output via PA9
// *************************************************************************
{
	SYSCFG->CFGR3 |= SYSCFG_CFGR3_PINMUX2_0;
	USART1->CR1 &= ~(USART_CR1_UE);	//disable USART
	USART1->CR1 |= USART_CR1_TE;//re-enable transmitter
	USART1->CR1 |= USART_CR1_UE;	//re-enable USART
	gTxOff = 0;
}

//time ops ################################################################

void op_XNOP(void) // *****************************************************
// ( delay -- )
// start a very short blocking delay
// duration is modified by stack parameter
// example for a minimal loop
// : TEST 0xA0 DO DUP OH 0 XNOP DUP OL 100 XNOP AGAIN;
// *************************************************************************
{
uint32_t loops;

	loops = *gfSP;
	gfSP--;	
	loops--;	
	while (loops) {
		loops--;
	}
}


void op_MS(void)  // *******************************************************
// ( milliSeconds -- )
//blocks CForth execution for *gfSP milliSeconds
// example
// 1000  MS  blocks for 1 second
// *************************************************************************
{
uint32_t tibstate;

	  gfMSCOUNT = *gfSP;		//milliseconds
	  gfSP--;
	  while (gfMSCOUNT)  {
		servSystem();
//serve Background process
		if (gfStepSpeed)  {
		   if ((gfUserHeader[4].toCompile > 0) ||
				(gfUserHeader[4].toCompile == 0xFFFFFFFE)) {
			  if (gfSTEPCOUNT == 0) subMOT();
			}
		}
		if (gfTermInput == CTRL_C) {
			gfTermInput = 0;
			return;
		}
		if (gfBackProcess && gfExecUser)  {
			tibstate = subQuery();
			if (tibstate == 1) backProcEval();
			if (tibstate == 2) gfErrors |= ERR_CANCEL;
		   if (gfErrors)  return;
		}
	}
}


void op_DECIS(void)  //	****************************************************
// ( deciSeconds -- )
//blocks CForth execution for *gfSP deciSeconds
// example
// 10 DCS blocks for 1 second
// *************************************************************************
{
	*gfSP *= 100;		//deciSeconds = milliSeconds * 100;
	op_MS();
}


void op_TIX(void)  // ******************************************************
// ( time -- )
// start a countdown measured in tenths of seconds.
// countdown is read out by TIME (non-blocking)
// *************************************************************************
{
	gfTIXCOUNT = *gfSP *100;	   // internally scaled in milliseconds
	gfSP--;
}


void op_TIME(void) // *****************************************************
// ( -- time)
//returns remaing time in tenths of seconds
// time coutdown  is not finished bx TIX read
//else returns 0 after countdown has finished
// *************************************************************************
{
	gfSP++;
	*gfSP = gfTIXCOUNT/100;
}


void op_FREQ(void) // ****************************************************
// ( freq -- )
//start peripheral TIMER2 to work on PA0 and/or PB1 as output
// freq	is the divisor, prescaled SYSCLK is divided by to get PWM frequency
// *************************************************************************
{
	if (gfExecUser) checkBackNum();

	gfFreq = *gfSP;		//standardized divisor, adjustment see below
	gfSP --;
	subFREQ();
	subPWM1();
	subPWM2();
}


void subFREQ(void) // ****************************************************
// ( freq -- )
//start peripheral TIMER1 to work on PA14(ch.1) and/or as output PA1(ch.2) 
// *************************************************************************
{

//internal divisor factor adjustment 
	if (gfFreq > 0x0000FFFF)  gfFreq = 0x0000FFFF;

		RCC->APBENR2 |= RCC_APBENR2_TIM1EN;

	if (gfFreq)  {										//start timer
		if ((TIM1->CR1 & TIM_CR1_CEN) == 0)  {	//not yet initialized
			TIM1->SR = 0;
			TIM1->CR1 = 0;
			TIM1->CCMR1 = (6<<12) |	(6<<4);
						TIM1->BDTR |= TIM_BDTR_MOE;	//essenital xtra against TIM2,3
#ifdef CLOCK_48M
			TIM1->ARR = 2000;		//max.prescaled timer input 24kHz
														//about 11 bit resolution
														//but easier calculation
														//freq=24 gives 1000Hz pulse
#else		//CLOCK_3M
			TIM1->ARR = 200;		//max.prescaled timer input 15kHz
														//about 7.5 bit resolution
														//but easier calculation
														//freq=15 gives 1000Hz pulse
#endif
			TIM1->CR1 |= TIM_CR1_CEN;
		}
//initial or restart of timer:
		TIM1->PSC = gfFreq - 1;
		//PSC is 16	bit register, Timer1 freq 24.0(15.0)kHz down to ca 0.1Hz
		//1 is added in TIM1_PSC prescaler divisor
		if (gfPWM1 < 0) TIM1->CCER &= ~TIM_CCER_CC1E;//disable output handling
		else TIM1->CCER	|= TIM_CCER_CC1E;	//enable output handling
		if (gfPWM2 < 0) TIM1->CCER &= ~TIM_CCER_CC2E;//disable output handling
		else TIM1->CCER	|= TIM_CCER_CC2E;	//enable output handling
	}
	else  {										//gfFREQ = 0 --> Timer OFF
		TIM1->CR1 &= ~TIM_CR1_CEN;		//disable timer
			gfPWM1 = -1;
		GPIOA->MODER &= ~(GPIO_MODER_MODE14);	//Set PA14 as input(default)
		GPIOA->PUPDR &= ~GPIO_PUPDR_PUPD14;
		GPIOA->PUPDR |= GPIO_PUPDR_PUPD14_1;	//pull-down
		TIM1->CCER &= ~TIM_CCER_CC1E;		//disable output handling
		gfPWM2 = -1;
		GPIOA->MODER &= ~(GPIO_MODER_MODE1);	//Set PA1 as input(default)
		GPIOA->PUPDR &= ~GPIO_PUPDR_PUPD1;
		GPIOA->PUPDR |= GPIO_PUPDR_PUPD1_1;	//pull-down
		TIM1->CCER &= ~TIM_CCER_CC2E;		//disable output handling
	}
}


void op_PWM1(void) // ******************************************************
// ( pwm -- )
//start peripheral TIMER1, channel 1 to work on PA14 as output
// *************************************************************************
{

	if (gfExecUser) checkBackNum();

	gfPWM1 = (int32_t)*gfSP;
	gfSP --;

	if ((TIM1->CR1 & TIM_CR1_CEN) == 0)  {	//timer emergency activation
#ifdef CLOCK_48M
		gfFreq = 24;	//1.0  kHz
#else
		gfFreq = 15;	//1.0  kHz
#endif
		subFREQ();
	}
	subPWM1();

}


void subPWM1(void) // *****************************************************
// ( pwm -- )
//start peripheral TIMER1, channel 1 to work on PA14 as output
// *************************************************************************
{

	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPD14;		//pull-up/down off,wasts current
	GPIOA->MODER &= ~(GPIO_MODER_MODE14);	//set PA14 temporarily as input
//handle the different I/O cases:
	if (gfPWM1 < 0)  {
		GPIOA->PUPDR |= GPIO_PUPDR_PUPD14_1;//pull-down=default
		TIM1->CCER &= ~TIM_CCER_CC1E;		//disable output handling
		return;
	}

	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPD14;		//pull-up/down off,wasts current

	if (gfPWM1 == 0)  {
		GPIOA->MODER |= GPIO_MODER_MODE14_0;// Set PortA14 as output
		GPIOA->BSRR = GPIO_BSRR_BR14;		// permanent LOW
		return;
	}
#ifdef CLOCK_48M
	if (gfPWM1 >= 2000)  {
		gfPWM1 = 2000;
		GPIOA->MODER |= GPIO_MODER_MODE14_0;// Set PortA14 as output
		GPIOA->BSRR = GPIO_BSRR_BS14;		// permanent HIGH
		return;
	}
#else
	if (gfPWM1 >= 200)  {
		gfPWM1 = 200;
		GPIOA->MODER |= GPIO_MODER_MODE14_0;// Set PortA14 as output
		GPIOA->BSRR = GPIO_BSRR_BS14;		// permanent HIGH
		return;
	}
#endif
	//set PA14 in alternate function mode
	GPIOA->MODER |= GPIO_MODER_MODE14_1;
	GPIOA->AFR[1] &= 0xF0FFFFFF;			//TIM1,CH1 @PA14 is AF10
	GPIOA->AFR[1] |= 0x0A000000;
	TIM1->CCER |= TIM_CCER_CC1E;			//enable output handling
	TIM1->CCR1 = (uint32_t)gfPWM1;			//H period ca.11 bit resolution
}


void op_PWM2(void) // ******************************************************
// ( pwm -- )
//start peripheral TIMER1, channel 2 to work on PA1 as output
// *************************************************************************
{

	if (gfExecUser) checkBackNum();

	gfPWM2 = (int32_t)*gfSP;
	gfSP --;

	if ((TIM1->CR1 & TIM_CR1_CEN) == 0)  {	//timer emergency activation
#ifdef CLOCK_48M
		gfFreq = 24;	//1.0  kHz
#else
		gfFreq = 15;	//1.0  kHz
#endif
		subFREQ();
	}
	subPWM2();

}


void subPWM2(void) // *****************************************************
// ( pwm -- )
//start peripheral TIMER1, channel 2 to work on PA1 as output
// *************************************************************************
{

//change NRST funtion to PA1 GPIO:
	  SYSCFG->CFGR3 |= SYSCFG_CFGR3_PINMUX1_1;	

		GPIOA->MODER &= ~(GPIO_MODER_MODE1);//Set PA1 temporarily as input
	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPD1;		//pull-up/down off,wasts current
	if (gfPWM2 < 0)  {
		GPIOA->PUPDR |= GPIO_PUPDR_PUPD1_1;	//pull-down = default
		TIM1->CCER &= ~TIM_CCER_CC2E;		//disable output handling
		return;
	}
	if (gfPWM2 == 0)  {
		GPIOA->MODER |= GPIO_MODER_MODE1_0;	// Set PortA1 as output
											//OTYPER as already config.
		GPIOA->BSRR = GPIO_BSRR_BR1;		// permanent LOW
		TIM1->CCER &= ~TIM_CCER_CC2E;		//disable output handling
		return;
	}
#ifdef CLOCK_48M
	if (gfPWM2 >= 2000)  {
		gfPWM2 = 2000;
		GPIOA->MODER |= GPIO_MODER_MODE1_0;	// Set PortA1 as output
											//OTYPER as already config.
		GPIOA->BSRR = GPIO_BSRR_BS1;		// permanent HIGH
		TIM1->CCER &= ~TIM_CCER_CC2E;		//disable output handling
		return;
	}
#else
	if (gfPWM2 >= 200)  {
		gfPWM2 = 200;
		GPIOA->MODER |= GPIO_MODER_MODE1_0;	// Set PortA1 as output
											//OTYPER as already config.
		GPIOA->BSRR = GPIO_BSRR_BS1;		// permanent HIGH
		TIM1->CCER &= ~TIM_CCER_CC2E;		//disable output handling
		return;
	}
#endif
	//set PA1 in alternate function mode
	GPIOA->MODER |= GPIO_MODER_MODE1_1;		//AF mode
	GPIOA->AFR[0] &= 0xFFFFFF0F;			//TIM1,CH2 @PA1 is AF5
	GPIOA->AFR[0] |= 0x00000050;
	TIM1->CCER |= TIM_CCER_CC2E;			//enable output handling
	TIM1->CCR2 = (uint32_t)gfPWM2;			 //H period ca. 11 bit resolution
}


void op_MOT(void)  // ******************************************************
// ( StepSpeed -- )
//starts background process, controlled by "system"user VARiable NSTP
//2-Phase pulse scheme, must be expanded externally to 4 phases and amplified
//2 drive ports P12 and PA13
//control of motor power must be handled externally
//StepSpeed=0:stepper off, negative value: rotation counter clockwise
//is counted down with 1ms clock (Arm SsyTick)
//gfUserHeader[4].toCompile is the actual number of steps to be performed.
//							is reset when number of steps were performed
//NSTP= 0:rotation stops, else number of steps to be performed
//NSTP= -2-->permanent rotation in stepper mode
//NSTP= -1:NORMAL (commutated) MOTOR drive (steady H Bridge, no steps)
//then: 
// if (stepSpeed >= 0) rotation clockwise, PWM1 is started with value of speed
// else rotation counter clockwise, PWM1 is started with abs(value) of speed
// *************************************************************************
{
	if (gfExecUser) checkBackNum();

	if (gfUserHeader[4].toCompile == 0xFFFFFFFF)  {	//signed int -1
		gfStepSpeed = 0;					//neutralize background process
		GPIOA->MODER &= ~(GPIO_MODER_MODE12|GPIO_MODER_MODE13);
		GPIOA->MODER |= (GPIO_MODER_MODE12_0|GPIO_MODER_MODE13_0);//outp
		if ((int32_t)*gfSP > 0)  {			//drive normal motor clockwise
			GPIOA->BSRR = GPIO_BSRR_BR12;
			GPIOA->BSRR = GPIO_BSRR_BS13;
		}
		else if ((int32_t)*gfSP < 0)  {		//drive motor counterClockwise
			*gfSP = -(*gfSP);
			GPIOA->BSRR =  GPIO_BSRR_BS12;
			GPIOA->BSRR =  GPIO_BSRR_BR13;
		}
		else  {	//==0						//includes set PWM1->0
			GPIOA->BSRR =  GPIO_BSRR_BR12;	//stop motor
			GPIOA->BSRR =  GPIO_BSRR_BR13;
		}
		op_PWM1();							//start PWM	@PB1 with value of TOS
											//PWM ratio limited to max 100
											//(ext.PWM-HW driver needed)
		return;
	}
	//else Stepper Mot.
	if (gfStepSpeed == 0)  {				//old value
		gfStepPhase = 0; 
		gfSTEPCOUNT = 0;					//restart new phase
//override user GPIO setup
		GPIOA->MODER &= ~(GPIO_MODER_MODE12|GPIO_MODER_MODE13);
		GPIOA->MODER |= GPIO_MODER_MODE12_0|GPIO_MODER_MODE13_0;//outp
	}
	gfStepSpeed = *gfSP;
	gfSP--;
	if (gfStepSpeed == 0)  {				//new value
										//switch motor power off externally!
		GPIOA->BSRR = GPIO_BSRR_BR12;	
		GPIOA->BSRR = GPIO_BSRR_BR13;
		gfUserHeader[4].toCompile = 0;
	}
	gfSTEPCOUNT = 0;					//restart new phase, trigger subMOT
	subMOT();							//countdown in SysTick_Handler()
}


void subMOT(void)  // ******************************************************
//handles next stepper step	in background process
//comment see above
// *************************************************************************
{
int32_t direction, nsteps;

//drive stepper motor
	nsteps = gfUserHeader[4].toCompile;		//user access: NSTP
	if ((nsteps == 0)||(nsteps == 0xFFFFFFFF)) return;
											//final or DC	motor mode

	direction = gfStepSpeed;
	if (direction >= 0)  {	//run clockwise
		gfSTEPCOUNT = gfStepSpeed;			//countdown in SysTick_Handler()
		switch (gfStepPhase)  {
		case 0:
			GPIOA->BSRR = GPIO_BSRR_BR12;
			GPIOA->BSRR = GPIO_BSRR_BR13;	//repetition essential
			break;
		case 1:
			GPIOA->BSRR = GPIO_BSRR_BS13;
			break;
		case 2:
			GPIOA->BSRR = GPIO_BSRR_BS12;
			break;
		case 3:
			GPIOA->BSRR = GPIO_BSRR_BR13;
			break;
		}
	}
	else  {									//run counter clockwise
		gfSTEPCOUNT = -gfStepSpeed;			//countdown in SysTick_Handler()
		switch (gfStepPhase)  {
		case 0:
			GPIOA->BSRR = GPIO_BSRR_BR12;	//repetition essential
			GPIOA->BSRR = GPIO_BSRR_BR13;
			break;
		case 1:
			GPIOA->BSRR = GPIO_BSRR_BS12;
			break;
		case 2:
			GPIOA->BSRR = GPIO_BSRR_BS13;
			break;
		case 3:
			GPIOA->BSRR = GPIO_BSRR_BR12;
			break;
		}
	}
	gfStepPhase ++;
	if (gfStepPhase >= 4)  gfStepPhase = 0;
	if (nsteps != 0xFFFFFFFE) gfUserHeader[4].toCompile = nsteps - 1;
				//update countdown , nsteps=0xFFFFFFFE=-2: permanent rotation
}


// peripheral ops ##########################################################
uint32_t checkValidIO(uint32_t mode)  //	********************************
//simplified test for Port
//1=PA1, 2=PA12, 3=PA13, 4=PA14, 5=PA8(Usart_Tx disabled)
// *************************************************************************
{
uint32_t pinNo;

	pinNo = *gfSP;
	gfSP--;
		
		if (mode && (pinNo == 0xFFFFFFFF))  return 0xFFFFFFFF;
		if ((pinNo == 0) ||	(pinNo > 5))  {
			gfErrors |= ERR_NUMVAL;	
			pinNo = 0;
		}
		if (pinNo == 5)  {
			if (SYSCFG->CFGR3 & SYSCFG_CFGR3_PINMUX2_0)  {
				SYSCFG->CFGR3 &= ~SYSCFG_CFGR3_PINMUX2_0;
				gTxOff = 1;
				USART1->CR1 &= ~(USART_CR1_UE);		//disable USART
				USART1->CR1 &= ~(USART_CR1_TE);		//disable transmitter
				USART1->CR1 |= USART_CR1_UE;		//re-enable USART
			}
			pinNo = 8;
		}
		if ((pinNo != 1) && (pinNo != 8)) pinNo += 10;
		return pinNo;
	}


	void checkPWM(uint32_t pinNo)  // ******************************************
	// *************************************************************************
	{

		if (pinNo == 1)  {
			if (gfPWM1 != -1)  {
				TIM1->CCER &= ~TIM_CCER_CC2E;	//disable output handling
				gfPWM1 = -1;					//PWM OFF
			}
		}
		if (pinNo == 14)  {
			if (gfPWM2 != -1)  {				//PWM2 is at PA14
				TIM1->CCER &= ~TIM_CCER_CC1E;	//disable output handling
				gfPWM2 = -1;					//PWM2 OFF
			}
		}
	}


	void op_PINOH(void)  //	*****************************************************
	// (Port/Pin-- )
	// sets GPIO to output HIGH	in Push/Pull mode
	// parameter: pin number: if Number Base= Hex:A0,A1,A4..A7,AF,B0,B3,B4,.,B8
	// else pin numbers must be entered with leading 0x, e.g. 0xA0,0xAA,0xB7
	// or as corresponding decimal number, e.g.	0xA0=160, 0xAA=170,0x B0=176
	//		more simple:subtract 160 from decimal value:e.g. PA15=15,PB0=16,PB8=24
	// specific	pin numbers are reserved for system or not implemented
	// *************************************************************************
	{
	uint32_t pinNo, pos, mask;

		pinNo = checkValidIO(1);
		if (gfErrors) return;

			checkPWM(pinNo);
			pos = pinNo << 1;
			mask = ~(3 << pos);
			GPIOA->PUPDR &= mask;				//pull-up/down off, wasts current
			GPIOA->MODER &= mask;
			GPIOA->MODER |= 1 << pos;
			GPIOA->BSRR = 1 << pinNo;
	}


	void op_PINOL(void)  //	****************************************************
	// (Port/Pin -- )
	// sets GPIO to output LOW
	// comment about pinNo see op_PINOH
	// *************************************************************************
	{
	uint32_t pinNo, pos, mask;

		pinNo = checkValidIO(1);
		if (gfErrors) return;

			  checkPWM(pinNo);
			  pos = pinNo << 1;
			  mask = ~(3 << pos);
			  GPIOA->PUPDR &= mask;				//pull-up/down off, wasts current
			  GPIOA->MODER &= mask;
			  GPIOA->MODER |= 1 << pos;
			  GPIOA->BSRR =  1 << (pinNo + 16);
	}


	void op_PINIPU(void)  // ****************************************************
	// (Port/Pin -- )
	// configures GPIO as input with internal pull-up resistor
	// comment about pinNo see op_PINOH
	// *************************************************************************
	{
	uint32_t pinNo, pos, mask;

		pinNo = checkValidIO(1);
		if (gfErrors) return;

			checkPWM(pinNo);
			pos = pinNo << 1;
			mask = ~(3 << pos);
			GPIOA->MODER &= mask;				//configured as input
			GPIOA->PUPDR &= mask;				//first HiZ
			GPIOA->PUPDR |= 1 << pos;			//now pull-up
	}


	void op_PINIPD(void)  // ***********************************************
	// (Port/Pin -- )
	// configures GPIO as input with internal pull-down resistor
	// comment about pinNo see op_PINOH
	// *************************************************************************
	{
	uint32_t pinNo, pos, mask;

		pinNo = checkValidIO(1);
		if (gfErrors) return;

			checkPWM(pinNo);
			pos = pinNo << 1;
			mask = ~(3 << pos);
			GPIOA->MODER &= mask;				//configured as input
			GPIOA->PUPDR &= mask;				//first HiZ
			GPIOA->PUPDR |= 2 << pos;			//now pull-down
	}


	void op_PINIZ(void)  //	****************************************************
	// (Port/Pin -- )
	// configures GPIO as HI-Z input
	// comment about pinNo see op_PINOH
	// *************************************************************************
	{
	uint32_t pinNo, pos, mask;

		pinNo = checkValidIO(1);
		if (gfErrors) return;

				pos = pinNo << 1;
				mask = ~(3 << pos);
				GPIOA->MODER &= mask;			//configured as input
				GPIOA->PUPDR &= mask;			//HiZ
	}


	void op_READDIG(void)  // *************************************************
	// (port -- digitalLevel)
	// returns digital level of addressed GPIO
	// pinNo is always accepted and returned NO	MATTER if port is implented in CPU
	// NO MATTER if port the is configured as digital input or reserved
	// comment about pinNo see op_PINOH
	// *************************************************************************
	{
	uint32_t pinNo, level;

		pinNo = checkValidIO(1);
		if (gfErrors) return;

		level = GPIOA->IDR & (1 << pinNo);
	//even if pin is not implemented by hardware
		gfSP++;
		if (level) *gfSP = 1;
		else  *gfSP = 0;
	}


	void op_READADC(void)  // **************************************************
	//(PortApin_number -- analogLevel)
	//returns analog level of addressed port. Initalizes pin for ADC automatically
	//PA1,PA12,PA13,PA14 and PA8 are supported
	//parameter: pin number 1,2,3,4,5, special:-1=0xFFFFFFFF
	// *************************************************************************
	{
	uint32_t port, mask;
	const uint8_t initStr[] = "  (ADC init) ";
	const uint8_t offStr[] = " (ADC off) ";

		port = checkValidIO(0);
		if (gfErrors) return;
		if (port == 0xFFFFFFFF)  {		//i.e. signed -1
			disableADC();
			sendString(offStr);
			return;
		}
	if ((RCC->APBENR2 & RCC_APBENR2_ADCEN) == 0)  {
	   initADC();
	   sendString(initStr);
	}
	ADC1->ISR = 0x000007FF;			//clear relevant all bits by writing 1

	mask = 3 << (port << 1);
		GPIOA->MODER |= mask;		//analog input
		GPIOA->PUPDR &= ~mask;		//HiZ
//		channel = port;
	ADC1->CHSELR = (1 << port);		//ADC_channel = port @ STM32C011

//result is improved by multiple ADC sampling
	mask = 0;
	for (port = 0; port < 4; port++) {	//make average of 4 samples
		ADC1->CR |= ADC_CR_ADSTART;
		while ((ADC1->ISR & ADC_ISR_EOC) == 0) ;
		mask += (uint32_t)ADC1->DR;
		wait5us();
	}
	gfSP++;
	*gfSP = mask >> 2;		//div/4
	if (mask & 0x00000002) (*gfSP)++;	//round up if remainder(div/4) > 1
/*
//alternative if fast result is needed:
	ADC1->CR |= ADC_CR_ADSTART;
	while ((ADC1->ISR & ADC_ISR_EOC) == 0) ;
	*gfSP= (uint32_t)ADC1->DR;
*/
}


void initADC(void) // ***************************************************
// ( -- )
// initalizes ADC in general
// once initialized, it will be active during the session.
// takes some extra current
// *************************************************************************
{
	RCC->APBRSTR2 &= ~(RCC_APBRSTR2_ADCRST);
	RCC->APBENR2 |= RCC_APBENR2_ADCEN;
//AVR voltage regulator starting time is regular 20us
	wait5us();					//as a precaution
	wait5us();
	wait5us();
	wait5us();
	ADC1->CFGR1 = 0;					//right aligned, 12bit resolution
 #ifdef CLOCK_48M	
	ADC1->CFGR2 = ADC_CFGR2_CKMODE_1;//CKMODE=10,clock source=APB2_CLK/4=12MHz
	ADC1->SMPR = 2;					//sampling time about 13.5 clock cycles
#else
	ADC1->CFGR2 = ADC_CFGR2_CKMODE_0;	//CKMODE=01,clock source=APB_CLK/2
	ADC1->SMPR = 0;						//sampling time about 1.5 clock cycles
#endif
	ADC->CCR = 0;						//no special ADC channels active
	ADC1->CR |= ADC_CR_ADEN;
	while ((ADC1->ISR & ADC_ISR_ADRDY) == 0) ;
	wait5us();
}


void disableADC(void)  // ***************************************************
// ( -- )
// initalizes ADC in general
// once initialized, it will be active during the session.
// takes some extra current
// *************************************************************************
{
	ADC1->ISR |= ADC_ISR_ADRDY;				//clear by writing 1 
	ADC1->CR |= ADC_CR_ADDIS;				//set ADDIS bit
	while ((ADC1->CR & ADC_CR_ADEN) != 0) {
		servSystem();
	}
	RCC->APBENR2 &= ~(RCC_APBENR2_ADCEN);
	RCC->APBRSTR2 |= RCC_APBRSTR2_ADCRST;
}


void op_INITSPI(void)  // ***************************************************
// (Mode(0,1,2,3)  Bits(4..32) Clock -- )  //3 stack parameter !!
// SPI is always Master
// SPI uses PA1(SCK),PA12(MOSI),PA13(MISO), 
// code is hand-written because STM	support for SPI	is bad
// *************************************************************************
{
	gfSpiClock = *gfSP;
	gfSP--;
	gfSpiBits = *gfSP;
	gfSP--;
	gfSpiMode = *gfSP;
	gfSP--;
	subInitSpi();
}

void subInitSpi(void)  // **************************************************
//mode0  (CPOL= 0, CPHA=0): clock idle low
//		data sampled on rising clk edge and shifted out at the falling edge
//mode1 (CPOL= 0, CPHA=1): clock idle low
//		data sampled on falling clk edge and shifted out at the rising edge
//mode2 (CPOL= 1, CPHA=0): clock idle high
//		data sampled on falling clk edge and shifted out at the rising edge
//mode3 (CPOL= 1, CPHA=1): clock idle high
//		data sampled on rising clk edge and shifted out at the falling edge
// *************************************************************************
{
	if ((gfSpiMode>3)||(gfSpiBits<4)||(gfSpiBits>32)||(gfSpiClock>0xFFFF)) {
		gfErrors |= ERR_NUMVAL;
		return;
	}
//PA1 is SPI1_SCK	configure as output push/pull
//PA12 is SPI1-MOSI, configure as output push/pull
//PA13 is SPI1_MISO	 configure as input pull down

	GPIOA->MODER &= ~(GPIO_MODER_MODE1|GPIO_MODER_MODE12|GPIO_MODER_MODE13);
											//all, especially PA13, are input
	GPIOA->MODER |= (GPIO_MODER_MODE1_0 | GPIO_MODER_MODE12_0);
											//PA1,PA12 push/pull out
	GPIOA->PUPDR &= ~(GPIO_PUPDR_PUPD13);
	GPIOA->PUPDR |= GPIO_PUPDR_PUPD13_1;	//PA13=MISO now pull-down
	if (gfSpiMode>=2) GPIOA->BSRR = GPIO_BSRR_BS1;//preset idle SCLK High
	else  GPIOA->BSRR = GPIO_BSRR_BR1;		//idle SCLK	Low
}


void op_SPI(void)  // ******************************************************
//(sendWord(or Byte) -- recvWord(or Byte))
//uses default parameters or user parameters if project is loaded from Flash,
//PA1 must be available (op_PA1SET), else a reset will happen
// /CS must be handled completely by the user, how ever
// *************************************************************************
{
uint32_t spiTX, spiRX;
int32_t k, loops;

	if (((GPIOA->MODER & 0x0F000000) != 0x01000000) ||
			  ((GPIOA->MODER & 0x0000000C) != 0x000000004)) subInitSpi();
									//i.e.PA1,PA12=output,PA13=input
									//emergency init with actual parameters

//for precaution, usually not necessary:
	if (gfSpiMode >= 2) GPIOA->BSRR = GPIO_BSRR_BS1;		//idle SCLK	High
	else  GPIOA->BSRR = GPIO_BSRR_BR1;						//idle SCLK	Low

	spiTX = *gfSP;
	spiRX = 0;
	k = gfSpiBits;

	do {
			k--;
			if (spiTX & (1<< k)) GPIOA->BSRR = GPIO_BSRR_BS12;	//MOSI high
			else GPIOA->BSRR = GPIO_BSRR_BR12;					//MOSI Low

			if ((gfSpiMode == 0) ||	(gfSpiMode == 3))  {
					GPIOA->BSRR = GPIO_BSRR_BR1;				//SCLK	Low
			}
			else GPIOA->BSRR = GPIO_BSRR_BS1;					//SCLK	High

			loops = gfSpiClock;
			while (loops) {
					loops--;
			}

			spiRX = spiRX << 1;					//MISO:shift temporary result
			spiRX |= (GPIOA->IDR >> 13) & 1;	//append next bit -->bit0

			if ((gfSpiMode == 0) ||	(gfSpiMode == 3))  {
					GPIOA->BSRR = GPIO_BSRR_BS1;				//SCLK	High
			}
			else GPIOA->BSRR = GPIO_BSRR_BR1;					//SCLK Low


			loops = gfSpiClock;
			while (loops) {
					loops--;
			}
	}  while (k);

	if (gfSpiMode >= 2) GPIOA->BSRR = GPIO_BSRR_BS1;		//idle SCLK High
	else  GPIOA->BSRR = GPIO_BSRR_BR1;						//idle SCLK Low
	* gfSP = spiRX;
}


void op_REGWRITE(void) // *************************************************
// (value  peripheral_register_address -- )
// writes "value" the addressed peripheral register
// register address: see Reference Manual and/or STM32G431xx.h
// *************************************************************************
{
uint32_t* RegAddr;

	if ((*gfSP < 0x40000000)||(*gfSP > 0x50001FFF)) {
										//BOT&TOP see RefMan
		gfErrors |= ERR_NUMVAL;
		return;
	}
	RegAddr = (uint32_t*)*gfSP;
	*RegAddr = *(gfSP-1);
	gfSP -= 2;
}


void op_REGREAD(void)  // **************************************************
// (peripheral_register_address -- value)
// returns the content of the addressed peripheral register
// register address: see Reference Manual and/or STM32G431xx.h
// *************************************************************************
{
uint32_t* RegAddr;

	if ((*gfSP < 0x40000000) ||	(*gfSP > 0x50001FFF)) {	
										//BOTTOM&TOP see RefMan
		gfErrors |= ERR_NUMVAL;
		return;
	}
	RegAddr = (uint32_t*)*gfSP;
	*gfSP = *RegAddr;
}

// structuring ops ########################################################

void op_IF(void)  // *******************************************************
//( -- addr )
//compiles doIF
//starts compilation of a IF ... ELSE ... THEN conditional
//At compile time,it compiles doIF, provides a placeholder for jump	address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOIF);
	prepareElse();
}


void op_CASE(void) // *******************************************************
//( -- addr )
//compiles doIF
//starts compilation of a CASE ... ELSE	...	THEN conditional
//At compile time,it compiles doCASE, provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOCASE);
	prepareElse();
}


void op_LTCASE(void)  // *************************************************
//( -- addr )
//compiles do<CASE
//starts compilation of a <CASE	...	ELSE ... THEN conditional
//At compile time,it compiles do<CASE,provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOLTCAS);
	prepareElse();
}


void op_GTCASE(void)  // *************************************************
//( -- addr )
//compiles do>CASE
//starts compilation of a >CASE	...	ELSE ... THEN conditional
//At compile time,it compiles do<CASE,provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOGTCAS);
	prepareElse();
}


void prepareElse(void) // *************************************************
//( -- addr )
//compiles common code of IF, CASE, <CASE
//At compile time,it compiles it provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	gfSP++;
	*gfSP = gfCodeIndex;		//keep in mind for ELSE	or THEN
	compileTCode(0xFF);
	compileTCode(0xFF);			//placeholder, will be filled by ELSE or THEN
}


void op_ELSE(void) // *****************************************************
//( addr -- addr)
//compiles doELSE
//compiles middle part of a IF (or CASE, <CASE)... ELSE	...	THEN conditional
//At compile time, it compiles doELSE ,its code address into the placeholder
//behind doIF and provides a placeholder for jump address.
//The stack entry with the code address provided by IF is
//exchanged with an entry that contains the code address of doELSE.
// *************************************************************************
{
uint32_t  temp, target;

	compileTCode(IDX_DOELSE);
	temp = gfCodeIndex;					//keep in mind for THEN
	compileTCode(0xFF);
	compileTCode(0xFF);			//placeholder, will be filled by THEN
	target = (uint32_t)(gfTCode + gfCodeIndex -1) & 0x0000FFFF;
	gfTCode[*gfSP] = (uint8_t)(target >> 8);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
							//write jump target into compilation of IF
//calculated 1 byte less because pointer is incremented in executeTCode()
	*gfSP = temp;
}


void op_THEN(void) // ****************************************************
//( addr -- )
//compiles termination of a IF ... ELSE	...	THEN conditional
//Essentially it compiles the terminating code address
//into the placeholder behind doIF (or doELSE if present).
// *************************************************************************
{
uint32_t target;

	target = (uint32_t)(gfTCode + gfCodeIndex -1);
	target &= 0x0000FFFF;
	gfTCode[*gfSP] = (uint8_t)(target >> 8);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
							//write jump target into compilation of IF or ELSE
							//word pointer index offset is bytes*4
//calculated 1 word less because pointer is incremented in executeTCode()
	gfSP--;
}


void op_DO(void)  // *******************************************************
//( -- addr )
//compiles doDO
//starts compilation of a DO ... WHILE or DO ... AGAIN loop.
//Essentially it provides a placeholder for the BREAK code address
//and sends its code address via stack to WHILE	or AGAIN
// *************************************************************************
{
	compileTCode(IDX_DODO);//at runtime DODO puts BREAK	adddress on Return Stack
	gfSP++;
	*gfSP = gfCodeIndex;		//sends loopback address to "WHILE" or "AGAIN"
	compileTCode(0xFF);
	compileTCode(0xFF);			//placeholder for BREAK	adddress
								//will be filled by WHILE or AGAIN
}

void op_UNTIL(void)  //	****************************************************
//( addr -- )
//compiles doUNTIL
//finishes compilation of a DO ... WHILE loop
//Essentially it compiles its code address for BREAK behind doDO
//and compiles the loopback code adress provided by DO behind doUNTIL.
// *************************************************************************
{
	compileTCode(IDX_DOUNTIL);
	prepareBack();
}


void op_WHILE(void)  //	****************************************************
//( addr -- )
//compiles doWHILE
//finishes compilation of a DO ... WHILE loop
//Essentially it compiles its code address for BREAK behind doDO
//and compiles the loopback code adress provided by DO behind doWHILE.
// *************************************************************************
{
	compileTCode(IDX_DOWHILE);
	prepareBack();
}


void op_AGAIN(void)  //	***************************************************
//(   -- )
//compiles doAGAIN
//finishes compilation of a DO ... AGAIN loop
//Essentially it inserts its code address for BREAK	compilation behind doDO
//and compiles the loopback code adress provided by DO behind doAGAIN.
// *************************************************************************
{
	compileTCode(IDX_DOAGAIN);
	prepareBack();
}


void prepareBack(void) // ********************************************************
// *************************************************************************
{
uint32_t target;

	target = (uint32_t)(gfTCode + *gfSP + 1);
	compileTCode((uint8_t)((target >> 8) & 0x000000FF));
	compileTCode((uint8_t)(target & 0x000000FF));//loopback address sent by DO
	target =  (uint32_t)(gfTCode + gfCodeIndex -1);
	gfTCode[*gfSP] = (uint8_t)((target >> 8) & 0x000000FF);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
//gfCodeIndex was incremented by "compileTCode()" ie. points to RPDEC then.
//must be decremented because gfIP is incremented when doBREAK etc.	is left
	compileTCode(IDX_RPDEC);		//compile ReturnStach corrections handler
	gfSP--;
}


void op_FOR(void)  // ******************************************************
//( -- addr )
//compiles doFOR
//starts compilation of a FOR ... LOOP loop
//The runtime behaviour is very similar to "C" programming:
//doFOR	expects 3 DataStack entries: startIndex, endIndex, Increment
//Increment can be positive or negative and can be greater than 1.
//For runtime details of the FOR...LOOP loop see doFOR and doLOOP below.
//Essentially FOR provides a placeholder for the BREAK code address
//and sends its code address via stack to LOOP.
// *************************************************************************
{
	compileTCode(IDX_DOFOR);//at runtime DOFOR puts BREAK adddress on Ret.Stack
	gfSP++;						//sends the return address to "LOOP"
	*gfSP = gfCodeIndex;
	compileTCode(0xFF);
	compileTCode(0xFF);			//placeholder, will be filled by LOOP
}


void op_LOOP(void) // *****************************************************
//( addr -- )
//compiles doLOOP
//finishes compilation of a FOR	...	LOOP loop
//Essentially it inserts its code address for BREAK	compilation behind doFOR
//and compiles the loopback code adress provided by FOR	behind doLOOP.
// *************************************************************************
{
uint32_t target;

	compileTCode(IDX_DOLOOP);
	target = (uint32_t)(gfTCode + *gfSP + 1);
	compileTCode((uint8_t)((target >> 8) & 0x000000FF));
	compileTCode((uint8_t)(target & 0x000000FF));//compile jump	back behind FOR
	target =  (uint32_t)(gfTCode + gfCodeIndex -1);
	gfTCode[*gfSP] = (uint8_t)((target >> 8) & 0x000000FF);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
									//sends the CONTI & BREAK address to "FOR"
//gfCodeIndex was incremented by "compileTCode()" ie. points to RP4DEC then.
//must be decremented here because gfIP is incremented when doBREAK	etc. is left
	compileTCode(IDX_RP4DEC);
	gfSP--;
}

void op_IDX(void)  // ******************************************************
//( -- index)
//at runtime= copies 2nd Return Stack entry = actual loop parameter on TOS
// *************************************************************************
{
	gfSP++;
	*gfSP = *(gfRP - 1);
}


void op_CONTI(void)  //	****************************************************
//( --- )
//initiates unconditional jump to behind doDO or doFOR (starts new loop)
//performs similar action as BREAK,
//but jumps to code behind (doDO+arg.) or (doFOR+arg.)
//More than one CONTI may be within a loop, each w.	individual decision
//gfIP will be incremented when op_DOCONTI is left
// *************************************************************************
{
uint8_t *cptr;
uint32_t temp;

	temp = (*gfRP - 2) | 0x20000000;
	cptr = (uint8_t*)temp;
	temp = *cptr;
	if (temp == IDX_DOLOOP)  gfIP = cptr - 1;
	else  {
		gfIP = cptr;
		op_DOAGAIN();
	}
}


void op_BREAK(void)  //	****************************************************
//( --- )
//initiates unconditional jump to (value from Return Stack)
// *************************************************************************
{
	gfIP = (uint8_t*)*gfRP;
		//the BREAK	IP is stored in *gfRP by doDO or doFOR
		//gfIP will be incremented in "executeTcode()" when op_BREAK is left
		//gfRP is reset by RP-1 (DO	loop) or RP-4 (FOR loop)
}


void op_ZBREAK(void)  // ****************************************************
//( flag --)
//if (flag==0) linear loop progression
//else it initiates a jump to (value from Return Stack)
// *************************************************************************
{
	if (*gfSP == 0) gfIP = (uint8_t*)*gfRP;		//perform BREAK	then
	//else do nothing
	gfSP--;
}


void op_RETURN(void)  // ***************************************************
//( --- )
//leaves a user thread before end (equivalent to "C" return;)
//see op_SEMIS , too
// *************************************************************************
{
	gfIP = (uint8_t*)*gfRP;
	if (gfRP > gfReturnStack) {
//new 31Dec23:
		if (gfBackProcess == 2)  {
			if (gfNestLevel == gfBackNestLevel)  {
				gfBackNestLevel = 0;
				gfBackProcess = 3;
			}
		}
		gfRP--;				//else gfRP==gfReturnStack-->finished
		gfNestLevel --;
	}
	else  {
		gfIP = 0;			// signal termination to executeTcode()
		if (gfBackProcess == 2)  gfBackProcess = 3;
	}
}


// compiler ops  ###########################################################

void op_PRINTSTACK(void)  // ***********************************************
// simple debug function: displays all stack entries. TOS is rightmost
//for debug purposes exceptionally it can be compiled into User Threads
// *************************************************************************
{
uint32_t k;
const uint8_t UnderflowStr [] = "underflow!	";
const uint8_t emptyStr [] = " --- ";

	sendTB('[');
	if (gfSP == (gfDataStack + 3)) sendString(emptyStr);
	else if (gfSP < gfDataStack + 3)  {
		sendString(UnderflowStr);
		gfSP = gfDataStack;		//re-init stack
		*gfSP=0;
		gfSP++;
		*gfSP = 0;				//four stack items as underflow reserve
		gfSP++;
		*gfSP = 0;
		gfSP++;
		*gfSP = 0;
	}
	else  {
		k = 4;
		while ((gfSP > gfDataStack + 3)&&(k <= (uint32_t)(gfSP-gfDataStack))) {
							//4 stack items as underflow reserve not displayed
			sendTB(SPACE);
			if (gfTempNumBase == 16)  send0X32(gfDataStack[k]);
			else  sendNS32(gfDataStack[k]);
			sendTB(SPACE);
			k++ ;
		}
	}
	sendTB(']');
	sendTB(SPACE);
}


void op_FLUSH()  //	********************************************************
// *************************************************************************
{
	gfSP = gfDataStack;
	*gfSP=0;
	gfSP++;
	*gfSP = 0;		//four stack items as underflow reserve
	gfSP++;
	*gfSP = 0;
	gfSP++;
	*gfSP = 0;	
}



void op_OPS(void)  // ******************************************************
//display all kernel operators: their names and their token values
// *************************************************************************
{
uint32_t k, n, p;
const uint8_t KernelStr [] = "Kernel:";

	sendString(KernelStr);
	sendTB(CRR);
	k = 1;			//don't show {ERROR} dummy name
	n = 0;
	while (k < KERNELSIZE) {
		sendTB(SPACE);
		for (p=0;p<9;p++)  {
			if (gfKernelNames[k][p]) sendTB(gfKernelNames[k][p]);
			else break;
		}
		while(p	< 7) {
			sendTB(SPACE);
			p++;
		}

		sendTB('#');
		sendX8((uint8_t)k);
		sendTB(SPACE);
		sendTB(SPACE);
		sendTB(SPACE);
		k++;
		n++;
		if (n==5) {
			sendTB(CRR);
			n = 0;
		}
	}
}


void op_USER(void) // *****************************************************
//display all user token: their names and their token values
// *************************************************************************
{
uint32_t k, p, wrap;

	k = 0;
	wrap = 0;
	sendTB(SPACE);
	while (k < gfNextNameCompi)  {
		if (k<10) sendTB(SPACE);
		sendNS32(k);
		sendTB(':');
		for (p=0;p<8;p++)  {
			if (gfUserHeader[k].name[p]) sendTB(gfUserHeader[k].name[p]);
			else break;
		}

		if (gfUserHeader[k].attributes & A_VARCON) {
			sendTB(0x28);	  //'('
			sendTB('V');
			sendTB('a');
			sendTB('r');
			sendTB('C');
			sendTB('o');
			sendTB('n');
			sendTB('=');
			if (gfTempNumBase == 16)  send0X32(gfUserHeader[k].toCompile);
			else sendNS32(gfUserHeader[k].toCompile);
			sendTB(SPACE);
			sendTB('@');
			send0X32(((uint32_t)&gfUserHeader[k].toCompile) & 0x0000FFFF);
			sendTB(0x29);	  //')'
			wrap += 2;
		}
		sendTB(SPACE);
		sendTB(SPACE);
		k++;
		wrap++;
		if (wrap >= 6) {
			sendTB(CRR);
			sendTB(SPACE);
			wrap = 0;
		}
	}
	if (wrap) sendTB(CRR);
}


void op_SEE(void)  // ******************************************************
//show the compiled body of a user thread
// *************************************************************************
{
uint32_t k, n, wrap, w, count, word, byte, address, index;
const uint8_t KernelStr [] = "can't SEE	Kernel Ops";
const uint8_t UserStr [] = "UserHeader_Struct=";
const uint8_t VarconStr [] = "VarCon:Value=";

	gfTokenStart = gfTibPos;

	scanNext();

	if (findUserToken() == 0) {	//updates "gfParsed"=index of "gfUserHeader"
		if (findKernelToken()) {
			sendString(KernelStr);
			return;
		}
		gfErrors |= ERR_STRINGTOKEN;
		return;
	}
	gfTokenStart = gfTibPos;
	sendString(UserStr);
	send0X32((uint32_t)gfUserHeader + gfParsed*16);	//16 bytes/struct
	sendTB(CRR);

	k = (gfUserHeader[gfParsed].toCompile - (uint32_t)gfTCode) & 0x00000FFF;

	wrap = 0;
	if ((gfUserHeader[gfParsed].attributes & A_VARCON) == 0) {
														//"normal" Function
		for (n = k;gfTCode[n] != IDX_SEMIS;n++) {
			wrap++;
			if (wrap >= 8) {
				sendTB(CRR);
				wrap = 0;
			}
			address = (uint32_t)(&(gfTCode[n])) & 0x0000FFFF;
			sendNX32(address);	//leading 0x2000.. and 0x suppressed
			sendTB(':');

			if (gfTCode[n] >= 0x80)  {					//new 18Jan25:
				index = gfTCode[n] & 0x7F;
				if (gfUserHeader[index].attributes & A_VARCON) {
					index2String(index);
					sendTB(0x28);	 //'('
					sendTB('=');
					if (gfTempNumBase==16) {
						send0X32(gfUserHeader[index].toCompile);
					}
					else  sendNS32(gfUserHeader[index].toCompile);
					sendTB(0x29);	 //')'
					sendTB(SPACE);
					sendTB(SPACE);
					continue;
				}
			}

			token2String(gfTCode[n]);
			if ((gfTCode[n]	>= IDX_DOIF) && (gfTCode[n]	<= IDX_DOLOOP)) {
				word = (uint32_t)(gfTCode[n+1] << 8) | (uint32_t)gfTCode[n+2];
				n += 2;
				sendTB(0x28);	 //'('
				sendNX32(word + 1);		//leading 0x suppressed
				//the compiled jump	address is 1 less than effectively executed
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOLIT8B) {
				n++;
				sendTB(0x28);	 //'('
				if (gfTempNumBase==16) send0X32((uint32_t)gfTCode[n]);
				else  sendNS32((uint32_t)gfTCode[n]);
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOLIT) {
				word = (uint32_t)(gfTCode[n+1] << 24);
				word |= (uint32_t)(gfTCode[n+2]	<< 16);
				word |= (uint32_t)(gfTCode[n+3]	<< 8);
				word |= (uint32_t)gfTCode[n+4];
				n += 4;
				sendTB(0x28);	 //'('
				if (gfTempNumBase==16) send0X32(word);
				else  sendNS32(word);
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOWVA) {
				n++;
				sendTB(0x28);	 //'('
				word  = (uint32_t)(gfTCode[n]);	//mod 01Jun21,bit7 not deleted
				index2String(word);
				word = (uint32_t)(&gfUserHeader[word].toCompile);
				sendTB('@');
				send0X32(word & 0x0000FFFF);	//leading 0x2000..	suppressed
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOSTR) {
				sendTB('"');
				n++;
				count = gfTCode[n];		//byte count, max.255!
				n++;
				while (count)  {
					byte  = gfTCode[n];
					n++;
					if ((byte >= 0x20) && (byte < 0x7F))  sendTB(byte);
					else  {
						sendTB(0x5C);		 //Backslash
						sendNibble((byte >> 4) & 0x0000000F);
						sendNibble(byte & 0x0000000F);
					}
					count --;
				}
				n-- ;	//during while{} already increment by for{ preempted
				sendTB('"');
				wrap += 3;
			}
			sendTB(SPACE);
			sendTB(SPACE);
		}
		address = (uint32_t)(&(gfTCode[n])) & 0x0000FFFF;
		sendNX32(address);	//leading 0x2000.. and 0x suppressed
		sendTB(':');
		sendTB(';');
	}
	if (gfUserHeader[gfParsed].attributes & A_VARCON) {
		sendString(VarconStr);
		if (gfTempNumBase == 16)  send0X32(gfUserHeader[gfParsed].toCompile);
		else  sendS32(gfUserHeader[gfParsed].toCompile);
		sendTB(SPACE);
		sendTB('@');
		send0X32((uint32_t)(&gfUserHeader[gfParsed].toCompile) & 0x0000FFFF);
						//address always in hex, leading 0x2000.. suppressed
	}
	sendTB(CRR);
}


void token2String(uint32_t tokenval)   // **********************************
// *************************************************************************
{
uint32_t k, p;

	if (tokenval < KERNELSIZE) {		//Kernel token
		for (p=0;p<8;p++)  {
			if (gfKernelNames[tokenval][p]) sendTB(gfKernelNames[tokenval][p]);
			else break;
		}
	}
	else {								//User token
		k = tokenval & 0x0000007F;
		for (p=0;p<8;p++)  {
			if (gfUserHeader[k].name[p]) sendTB(gfUserHeader[k].name[p]);
			else break;
		}
	}
}


void index2String(uint32_t index) // *************************************
//reconstruct VARCON name from address of .toCompile  struct entry
// *************************************************************************
{
uint32_t p;
uint8_t letter;
	
	sendTB(0x27); // = '
	for (p=0;p<8;p++)  {
		letter = gfUserHeader[index].name[p];
		if (letter) sendTB(letter);
		else break;
	}
	sendTB(0x27);
}


void op_MEM(void)  // *****************************************************
//(addr -- )
//send 128 byte entries starting from "addr" (16bit, w.o. leading 0x2000..)
//for debug purposes exceptionally it can be compiled into User Threads
//due to "little endian", native 32 bit entries (gfUserHeader[].toCompile, 
//and gfUserHeader[].Attributes) are displayed in reverse byte order
// *************************************************************************
{
uint8_t *address;
uint8_t value;
uint32_t k, wrap;
const uint8_t StartStr [] = "128 SRAM bytes starting from 0x";

	if (gfExecUser) checkBackNum();

	address = (uint8_t *)(0x20000000 | *gfSP);
	gfSP--;
	if ((uint32_t)address > 0x20007F80) {
		gfErrors |= ERR_NUMVAL;
		return;
	}
	sendTB(CRR);
	sendString(StartStr);
	sendX32((uint32_t)address);

	wrap = 15;
	for (k=0; k<128; k++)  {
		value = *(address + k);
		wrap++;
		if (wrap == 16) {
			sendTB(CRR);
			sendX32((uint32_t)(address + k));
			sendTB(':');
			wrap = 0;
		}
		sendX8(value);
		if ((wrap & 0x00000003) == 3) sendTB('|');
	}
	sendTB(CRR);
}


void op_STEP(void) // ****************************************************
//( --- )
// activates single step execution, trigger next step with TAB key
// special case: key "ESC" terminates execution at once (leave endless loops)
//for debug purposes exceptionally it can be compiled into User Threads
// *************************************************************************
{
	 gfBackProcess = 3;
}


void op_RUN(void)  // ****************************************************
//( --- )
//default state of background process, always active after system start 
//de-activates single step execution, but keeps the background process active
// special case: key "ESC" terminates execution at once (leave endless loops)
//for debug purposes exceptionally it can be compiled into User Threads
// *************************************************************************
{
	gfBackProcess = 1;
}


void op_NOBAK(void)  //	****************************************************
//( --- )
// de-activates background process.	Token Code speed is approx 40% faster
//for speed control exceptionally it can be compiled into User Threads
// *************************************************************************
{
	gfBackProcess = 0;
}


void op_BAKOP(void)  //	****************************************************
//( --- )
// restores RUN mode when background process is OFF
// *************************************************************************
{
const uint8_t backStr [] = " ->BackOP ";
	if (gfBackProcess == 0)  {
		sendString(backStr);
		gfBackProcess = 1;
	}
	//else dont't repeat
}


void op_FORGET(void)  // **************************************************
//delete the argument user function and all later defined ones
//example:
// : XYZ .......
// FORGET XYZ
// *************************************************************************
{
uint32_t k;

	gfTokenStart = gfTibPos;
	scanNext();
	if (findUserToken() == 0) {
		gfErrors |= ERR_USERSTRING;
		return;
	}
	gfTokenStart = gfTibPos;

//update compiler pointers
#ifdef MOTOR
	if (gfParsed < 5)  {		//NSTP, V1,2,3,4 cannot be deleted
#else
	if (gfParsed < 4)  {		//V1,2,3,4 cannot be deleted
#endif
		gfErrors |= ERR_CONTEXT;
		return;
	}
	gfNextNameCompi = gfParsed;
	gfUserHeader[gfParsed].name[0] = 0;
	gfCodeIndex = gfUserHeader[gfParsed].attributes & 0x00000FFF;
						//works with VARCON too, doesn't modify gfCodeIndex

	if ((gfAuto - 1) >= gfParsed) gfAuto = 0;	//new 05Apr25
//as precaution, set the rest of gfUserHeader[]	and gfTCode completely to zero
	initTop();
}


void op_ABORT()  //	********************************************************
//warm start. Re-initalizes runtime parameters and stacks,
//			  but keeps variable values and compiled functions
// *************************************************************************
{
uint32_t k;

//reconstruct project from data loaded from "popFlashProject()"
	gfComment = 0;
	gfCompiState = 0;
	gfExecUser = 0;
	gfBackNumValid = 0;
	gfNestLevel = 0;
	if (gfBackProcess) gfBackProcess = 1;	//bugfix 26Jan25
	gfTermCheck = 0;
	gfTempNumBase = gfNumBase;

	for (k = 0;	k < MAXUSER; k++)  {	//clear thread of new token code
		if (gfUserHeader[k].name[0] == 0) break;
	}
	gfNextNameCompi = k;
	if ((gfAuto - 1) >= k) gfAuto = 0;		//new 05Apr25

	k = CODETOP;
	while (1) {
	   if (gfTCode[k] == IDX_SEMIS) break;
	   if (k == 0) break;
	   k--;
	}
	

	if (k > 0) gfCodeIndex = k + 1;
	else gfCodeIndex = 0;

//as precaution, set the rest of gfUserHeader[]	and gfTCode completely to zero
	initTop();

	gfRP = gfReturnStack;
	*gfRP = 0;				//repeat as precaution,is essential when compiling
	gfIP = gfTCode;
	gfErrors = 0;
	op_FLUSH();
}


void initTop(void) // *******************************************************
// **************************************************************************
{
uint32_t k;
uint32_t *pointer;

	gfCodeStartIndex = gfCodeIndex;

	pointer = (uint32_t*)&(gfUserHeader[gfNextNameCompi]);
	while (pointer < (uint32_t*)&gfUserHeader[MAXUSER])  {
		*pointer = 0;
		pointer++;
	}
	for (k = gfCodeIndex; k < CODETOP; k++)  {
		gfTCode[k] = 0;
	}
}


void op_NRSET(void)  //	*****************************************************
//set pin4 to NRST funtion
//cases a ystemReset after been finished
// **************************************************************************
{
uint32_t newOptByte;

	SYSCFG->CFGR3 &= ~(SYSCFG_CFGR3_PINMUX1);
	newOptByte = FLASH->OPTR;
	newOptByte |= FLASH_OPTR_NRST_MODE_0;
	handleOptByte(newOptByte);
}


void op_PA1SET(void)  //	************************************************
//set pin4 to NRST funtion
// **************************************************************************
{
uint32_t newOptByte;

	SYSCFG->CFGR3 |= SYSCFG_CFGR3_PINMUX1_1;
	newOptByte = FLASH->OPTR;
	newOptByte &= ~FLASH_OPTR_NRST_MODE_0;
	handleOptByte(newOptByte);
}


void handleOptByte(uint32_t toHandle)  //	*********************************
//set pin4 to NRST funtion or PA1
//cases a ystemReset after been finished
// **************************************************************************
{
#define KEY1    0x45670123				//mandatory unlock key defined by STM
#define KEY2    0xCDEF89AB
	while ((FLASH->SR &FLASH_SR_BSY1) !=0);	//wait until no operation going on
	if ((FLASH->CR & FLASH_CR_LOCK) != 0) {
		FLASH->KEYR = KEY1;
		FLASH->KEYR = KEY2;
	}
#define OPTKEY1  0x08192A3B
#define OPTKEY2  0x4C5D6E7F
	while ((FLASH->SR & FLASH_SR_BSY1) != 0);	//wait until no op. going on
//unlock option bytes
	FLASH->OPTKEYR = OPTKEY1;
	FLASH->OPTKEYR = OPTKEY2;

	while ((FLASH->SR & FLASH_SR_BSY1) != 0);	//wait until no op. going on
	FLASH->OPTR = toHandle;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0);	//wait until no op. going on
	FLASH->CR	|= FLASH_CR_OPTSTRT;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0);	//wait until no op. going on
	msDelay(2);
	FLASH->CR	|= FLASH_CR_OBL_LAUNCH;			//causes C011 SystemReset !
}


void op_REPLACE(void)  // ***************************************************
// replace the token code of <target name> with that of <supplier name>
// Syntax: REPLACE	 <supplier name>  <target name>
// ( -- )  no stack parameters
// <supplier name> is always the first hit searched from TOP of userToken[]
// <target name> is always the first hit searched from TOP of userToken[]
// only token code between supplier and target is replaced
// If supplier and target have the same name,
// the search for target is continued down towards bottom of userToken[].
// original state is restored with "RESTORE"
//
// This PROCEDURE WORKS ONLY 
// on supplier, which is the first hit
// and target which is the first hit of search from TOP of userToken
// AND supplier is newer than target.
//
// If intended target is NEWER than supplier,a "retro" substitution is made
// as follows: compile a new UserToken with the code of old supplier.
//
// REPLACE inserts the supplier code address in gfUserHeader[target].toCompile
// but keeps the original gfTode index in gfUserHeader[target].attributes
// in 8bit token code format,a UserToken is the index in gfUserHeader[]	|=0x80
// *************************************************************************
{
uint32_t   supIndex, targIndex;
uint32_t   supToken, targToken;

//parse <supplier name> from TIB
	supIndex = getToken();			//index in gfUserHeader[] !!
	if (supIndex == 0xFFFFFFFF)  {
		gfErrors |= ERR_USERSTRING;
		return;
	}
	if (gfUserHeader[supIndex].attributes & A_VARCON)  {
		gfErrors |= ERR_CONTEXT; //replacement of CONST or VAR makes no sense
		return;
	}
//parse <target name> from TIB
	targIndex =  getToken();		//index in gfUserHeader[] !!
	if (targIndex == 0xFFFFFFFF)  {
		gfErrors |= ERR_USERSTRING;
		return;
	}
	if (gfUserHeader[targIndex].attributes & A_VARCON) {
		gfErrors |= ERR_CONTEXT; //replacement of CONST	or VAR makes no sense
		return;
	}

	if (supIndex <= targIndex) {
		gfErrors |= ERR_CONTEXT;
		return;
	}
	gfUserHeader[targIndex].toCompile = gfUserHeader[supIndex].toCompile;
}


void op_RESTORE(void)  // **************************************************
// Syntax: RESTORE	<original name>
// ( -- )  no stack parameters
// <original name> is always the first hit searched from TOP of userToken[].
// If this is "REPLACE"d, the threaded code is reset to the original, but only
// up in TCode[] to the start of the "native" code with "REPLACE"d token value.
// When it seems,that the code of the found token was not "REPLACE"d,the search
// is continued down to a possible second entry with same name in userToken[].
// If nothing has been "REPLACE"d, nothing gets changed
// The procedure is made possible as follows:
// REPLACE inserts the supplier code address in gfUserHeader[target].toCompile
// but keeps the original gfTode index in gfUserHeader[target].attributes
// in 8bit token code format,a UserToken is the index in gfUserHeader[]	|=0x80
// *************************************************************************
{
 uint32_t  index, actualCode, origCode;

//parse <name of originalized token> from TIB
	index = getToken();							//index in gfUserHeader[]
	if (index == 0xFFFFFFFF)  {
		gfErrors |= ERR_USERSTRING;
		return;
	}
	actualCode = gfUserHeader[index].toCompile;	//has been REPLACEd
	origCode = (uint32_t)((gfUserHeader[index].attributes & 0xFFF) + gfTCode);
												//=real code addresss in SRAM
//protect "native actualCode"
	if (actualCode < origCode) {
		gfErrors |= ERR_CONTEXT;
		return;
	}
	gfUserHeader[index].toCompile = origCode;	//restore in gfUserHeader[]
}


uint32_t getToken(void)   // ***********************************************
// *************************************************************************
{
int32_t k;
uint32_t n;
uint8_t refChar, userChar;

	gfTokenStart = gfTibPos;
	scanNext();
	gfTokenStart = gfTibPos;
	k = gfNextNameCompi	- 1;		//k is index of last compiled USER Token
									//search DOWN from the LAST	COMPILED

	while (k != -1)  {
		for (n=0; n < 16;n++) {
			userChar = gfUserHeader[k].name[n];
			refChar = gfTokenstr[n];
//transform into capital letters:
			if ((refChar >= 'a')&&(refChar <= 'z')) refChar -= 0x20;
			if (userChar != refChar)  break;
			if (userChar ==0) return k;//="gfUserHeader[]" index of found token
		}
		k--;
	}
	return 0xFFFFFFFF;					//if named token not found
}


void op_COLON(void)  //	****************************************************
//( --- )
// start the compilation of a user function.
// *************************************************************************
{
uint32_t  userSymbol;

	userSymbol = compileUserSymbol(0);
	if (userSymbol == 0) return;	

//bugfix 30Dec23:
	if (userSymbol == 2)  gfUserHeader[gfNextNameCompi].name[0] = 0;	
							//compilation cancelled, mark end of Symbol Table
	else  {
//will be a "normal" token, no VARCON
		gfCodeStartIndex = gfCodeIndex;
		gfCompiState = 1;
	}
	return;
}


void op_SEMIS(void)  //	*****************************************************
//( --- )
//terminates every compilation, i.e. must be the last thread token
// **************************************************************************
{
	if (gfCompiState != 0) {						//during compilation
		compileTCode(IDX_SEMIS);
		gfCompiState = 0;
		gfNextNameCompi	++;
		gfUserHeader[gfNextNameCompi].name[0] = 0;	//mark end of Symbol Table
//update compiler pointers
		gfCodeStartIndex = gfCodeIndex;
	}
	else  {										//interpreting (runtime) state
		gfIP = (uint8_t*)*gfRP;
		if (gfRP > gfReturnStack) {
//new 31Dec23:
			if (gfBackProcess == 2)  {
				if (gfNestLevel == gfBackNestLevel)  {
					gfBackNestLevel = 0;
					gfBackProcess = 3;
				}
			}
			gfRP--;						//else gfRP==gfReturnStack-->finished
			gfNestLevel --;
		}
		else  {
			gfIP = 0;				// signal termination to executeTcode()
			if (gfBackProcess == 2)  gfBackProcess = 3;
		}
	}
	gfTempNumBase = gfNumBase;
	return;
}


void op_DOSTRING(void) // *************************************************
//( --- )
//runtime operator, compiled by ."
//In ThreadCode memory it is followed by the bytes to be transmitted.
//differing from initial CForth versions, the string is "byte counted".
// *************************************************************************
{
uint32_t count;
uint8_t *nextIP;

	gfIP++;
	count = *gfIP;
	nextIP = gfIP + count;
	while (count)  {
		gfIP++;
		sendTB(*gfIP);
		count --;
	}
	gfIP = nextIP;
}


void op_DOLIT(void)  //	****************************************************
//( -- 32bitNumber)
//In the compiled thread, doLIT	puts the content of the next ThreadCode entry
//(=number value) on TOS at runtime.
// *************************************************************************
{
uint32_t word;

	gfSP++;
	gfIP++;
	word = (uint32_t)(*gfIP << 24);
	gfIP++;
	word |= (uint32_t)(*gfIP << 16);
	gfIP++;
	word |= (uint32_t)(*gfIP << 8);
	gfIP++;
	word |= (uint32_t)*gfIP;
	*gfSP = word;
}


void op_DOLIT8B(void)  // ****************************************************
//( -- 8bitNumber)
//In the compiled thread, doLIT	puts the content of the next ThreadCode entry
//(=number value) on TOS at runtime.
// *************************************************************************
{
	gfSP++;
	gfIP++;
	*gfSP = (uint32_t)*gfIP;
}


void op_DOVALUE(void)  // ***************************************************
//( -- value)
//In the compiled thread, the gfUserHeader[] index is compiled behind doVALUE.
//At runtime, it puts the word content of "gfUserHeader[].toCompile" on TOS.
//Referencing the VARCON by address improves modification at runtime
//different from G4xx version to save code length !!
//compileTCode() compiles only 8bit token, index is stored instead of address
// *************************************************************************
{
uint32_t index;

	gfSP++;
	gfIP++;
	index = (uint32_t)(*gfIP);				//mod 01Jun21,bit7 not deleted
	*gfSP = gfUserHeader[index].toCompile;	//special detour for "SEE"
}


void op_DOWVA(void)  //	****************************************************
//(value -- )
//In the compiled thread,the "value" address of VARCON is comiled behind doWVA
//At runtime, "value" is written into this ".toCompile" field.
//different from G4xx version to save code length !!
//compileTCode() compiles only 8bit token, index is stored instead of address
// *************************************************************************
{
uint32_t index;
uint32_t* address;

	gfIP++;
	index = (uint32_t)(*gfIP);				//& 0x7F  deleted 01June21
	address = &gfUserHeader[index].toCompile;
	*address = *gfSP;
	gfSP--;
}


void op_DOIF(void) // *****************************************************
//( flag --)
//compiled by IF
//takes a flag from TOS, if == 0, a jump is initiated:
//behind doELSE	if present,
//or to code address compiled by THEN
//if != 0, execution is performed until doELSE.
//or continues linear (if no doELSE)
// *************************************************************************
{
uint32_t jump;

	if (*gfSP != 0)  gfIP += 2;
	else  toElse();
	gfSP--;
}


void op_DOCASE(void)  // *****************************************************
//( reference --)
//compiled by CASE
//compare reference with NOS
//if equal a jump is initiated behind doELSE if present,
//or to code address compiled by THEN
//if != 0, execution is performed until doELSE.
//or continues linear (if no doELSE)
//!! after this operation former NOS is TOS now for furter handling. 
//!! DROP if not needed
// *************************************************************************
{
uint32_t temp, jump;
	
	temp = *(gfSP-1);
	if (temp == *gfSP) gfIP += 2;
	else  toElse();
	gfSP--;									//drop reference value
}


void op_DOLTCAS(void)  // ***************************************************
//( reference --)
//compiled by <CASE
//compare reference with NOS
//if NOS is less than reference execution is performed until doELSE.
//			or continues linear (if no doELSE)
//if greater or equal, a jump is initiated behind doELSE if present,
//			or to code address compiled by THEN
//!! after this operation former NOS is TOS now for furter handling. 
//!! DROP if not needed
// *************************************************************************
{
int32_t temp, jump;
	
	temp = (int32_t)*(gfSP-1);
	if (temp < (int32_t)*gfSP) gfIP += 2;
	else  toElse();
	gfSP--;								//drop reference value
}


void op_DOGTCAS(void)  // ***************************************************
//( reference --)
//compiled by >CASE
//compare reference with NOS
//if NOS is less than reference execution is performed until doELSE.
//			or continues linear (if no doELSE)
//if greater or equal, a jump is initiated behind doELSE if present,
//			or to code address compiled by THEN
//!! after this operation former NOS is TOS now for furter handling. 
//!! DROP if not needed
// *************************************************************************
{
int32_t temp, jump;
	
	temp = (int32_t)*(gfSP-1);
	if (temp > (int32_t)*gfSP) gfIP += 2;
	else  toElse();
	gfSP--;								//drop reference value
}


void toElse(void)  // *****************************************************
//common part of doIF,doCASE,doLTSCAS
// *************************************************************************
{
uint32_t jump;
	
	gfIP++;
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	gfIP = (uint8_t*)(0x20000000 | jump);	//word pointer increment =4bytes!
											//drop reference value
}


void op_DOELSE(void)  // ****************************************************
//( --- )
//compiled by ELSE
//if the flag of doIF != 0, doELSE manages a jump
//behind the code of the ELSE part (compiled by THEN).
//if the flag of doIF == 0, execution is continued behind doELSE.
//At the end of the ELSE part, execution is continued linar.
// *************************************************************************
{
uint32_t jump;

	gfIP++;
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	gfIP = (uint8_t*)(0x20000000 | jump);	//word pointer increment =4bytes!
}


void op_DODO(void) // *****************************************************
//( --- )
//compiled by DO
//doDO puts the BREAK code address on the ReturnStack
// *************************************************************************
{
uint32_t jump;

	gfRP++;			//prepare Return Stack for doUNTIL, doWHILE	or doAGAIN
	gfIP++;			//address next index of gfTCode
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	*gfRP = 0x20000000 | jump;
// *gfIP contains the IP jump offset relative to gfTCode base address
//this is adjusted for jump	to op_RPDEC.
//CONTI	needs 2 less go to doUNTIL,doWHILE,doAGAIN
//the jump address is compiled 1 less,
//because it is incremented by executeTCode() when BREAK etc. is leaving there
}

void op_DOUNTIL(void)  // **************************************************
//( flag ---)
//compiled by UNTIL
//doUNTIL takes the flag from TOS.
//If != 0, execution goes forward behind RP-1 for linear progression
//If == 0, execution loops back to behind (doDO	+ 1word)
// *************************************************************************
{
	if (*gfSP == 0)  op_DOAGAIN();
	else {
		gfRP--;					//remove address for CONTI or BREAK
//			  gfIP += 2;						//proceed behind RP1DEC
//bugfix 19Dec23:
			gfIP += 3;			//proceed behind doUNTIL parameter and RP1DEC
	}
	gfSP--;
}


void op_DOWHILE(void)  // **************************************************
//( flag ---)
//compiled by WHILE
//doWHILE takes the flag from TOS.
//If == 0, execution goes forward behind RP-1 for linear progression
//If != 0, execution loops back to behind (doDO	+ 1word)
// *************************************************************************
{
	if (*gfSP != 0)  op_DOAGAIN();
	else {
		gfRP--;					//remove address for CONTI or BREAK
//			  gfIP += 2;						//proceed behind RP1DEC
//bugfix 19DEC23:
			gfIP += 3;			//proceed behind doWHILE parameter and RP1DEC
	}
	gfSP--;
}


void op_DOAGAIN(void)  // ***************************************************
//( --- )
//compiled by AGAIN
//execution loops always back to behind (doDO + 1word)
// *************************************************************************
{
uint32_t jump;

	gfIP++;
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	gfIP = (uint8_t*)(0x20000000 | jump);	//jump behind (doDO	+ 1word)
											//word pointer increment =4bytes!
}


void op_DOFOR(void)  //	****************************************************
//( startIndex, endIndex increment -- )
//is RUNTIME, compiled by FOR
//3 items are copied from the DataStack to the ReturnStack:
//TOS is the increment/decrement of loop index per loop
//NOS is the loop end index for loop termination
//3rd DataStack entry -> 2nd ReturnStack entry:start index=loop	index
//Furthermore the BREAK	code address is put on top of the ReturnStack.
// *************************************************************************
{
uint32_t jump;
//put items to Return Stack:
	gfRP += 4;
	*(gfRP - 1) = (int32_t)*(gfSP - 2);	//actual loop counter (==start index)
	*(gfRP - 2) = (int32_t)*(gfSP -1);	//loop end index (for op_DOLOOP)
	*(gfRP - 3) = (int32_t)*gfSP;	//loop index inc/decrement(for op_DOLOOP)
	gfSP -= 3;
	gfIP++;					//address next index of gfTCode
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	*gfRP = 0x20000000 | jump;
//this is adjusted for jump	to op_RP4DEC. CONTI	needs 2 less go to doLOOP
//the jump address is compiled 1 less,
//because it is incremented by executeTCode() when BREAK etc. is leaving there
}


void op_DOLOOP(void)  // ***************************************************
//( --- )
//compiled by LOOP
//counts the 2nd ReturnStack entry up or down (depends on 4th entry),
//compares the 2nd entry with the 3rd one,
//and intiates loopback to behind doFOR	(if compare is in range)
//or execution goes forward behind RP-4 for linear progression.
// *************************************************************************
{
int32_t index, endIndex, deltaIndex, jump;

	index = *(gfRP - 1);
	endIndex = *(gfRP - 2);
	deltaIndex = *(gfRP - 3);
//	  index += deltaIndex;					//changed 20April24

	if (deltaIndex >= 0)  {
		if (index < endIndex)  {
			index += deltaIndex;			//changed 20April24
			*(gfRP - 1) = index; 
			gfIP++;							//address next index of gfTCode
			jump = (uint32_t)(*gfIP << 8);
			gfIP++;
			jump |= (uint32_t)*gfIP;
			gfIP = (uint8_t*)(0x20000000 | jump);	//jump behind doFOR
		}
		else  {
			gfRP -= 4;						//tidy up Return Stack
//bugfix 19DEC23:
			gfIP += 3;			//proceed behind doLOOP parameter and RP4DEC
		}
	}
	else  {
		if (index > endIndex)  {
			index += deltaIndex;			//changed 20April24
			*(gfRP - 1) = index;
			gfIP++;							//address next index of gfTCode
			jump = (uint32_t)(*gfIP << 8);
			gfIP++;
			jump |= (uint32_t)*gfIP;
			gfIP = (uint8_t*)(0x20000000 | jump);	//jump behind doFOR
		}
		else  {
			gfRP -= 4;						//tidy up Return Stack
//bugfix 19DEC23:
			gfIP += 3;			//proceed behind doLOOP parameter and RP4DEC
		}
	}
}


void op_RPDEC(void)  //	****************************************************
//( --- )
//compiled by WHILE	and AGAIN
//At runtime it removes the top	entry from the ReturnStack
// *************************************************************************
{
	gfRP--;
}


void op_RP4DEC(void)  // ***************************************************
//( --- )
//compiled by LOOP
//At runtime it removes tfour top entries from the ReturnStack
// *************************************************************************
{
	gfRP -= 4;
}

// sytem ops  ##############################################################

void op_AUTO(void) // *****************************************************
//( --- )
//Syntax: AUTOEXE <User Thread name>
//The specified user thread is automatically executed at system start.
//To get it available at system start, the system must start from Flash.
//For details see op_LOAD and op_SAVE
// *************************************************************************
{
	gfTokenStart = gfTibPos;

	scanNext();
	if (findUserToken() == 0) {
		if (gfTokenstr[0] == 0x30) gfAuto = 0;
		else  gfErrors |= ERR_USERSTRING;
		return;
	}
	gfAuto = gfParsed + 1;			//+1 to distinguish from no_AUTOEXE == 0
}


void op_BAUD(void) // *****************************************************
// (baud -- )
// parameters: ONLY FIRST 2 DIGITS of decimal baudrate are accepted:
// 57=57600,38=38400,19=19200,96=9600.Max 57600Baud is default.
// BaudRate input is always interpreted DECIMAL, independent of number base
// *************************************************************************
{
uint32_t temp;
const uint8_t nextBaudStr [] ="Active at next System Start";

	temp = *gfSP;
	gfSP--;
	if (temp == 0x19) temp = 19;
	if (temp == 0x38) temp = 38;
	if (temp == 0x57) temp = 57;
	if (temp == 0x96) temp = 96;

	if ((temp == 19)||(temp == 38)||(temp == 57)||(temp == 96))  {
		sendString(nextBaudStr);
		sendTB(CRR);
		gBaud = temp;
		pushFlashProject();
		gBaudSet = 0;			//mark baud change pending
	}
	else  gfErrors |= ERR_NUMVAL;
}


void op_QUEST(void)  //	*****************************************************
// ( -- )
//	displays a list of active system coonfiguration
// **************************************************************************
{
uint32_t k, r, q;
const uint8_t stackStr [] = "Stack=";
const uint8_t threadStr [] = "CodeThread=";
const uint8_t autoStr [] = "   AUTOEXE=";
const uint8_t noneStr [] = "none";
const uint8_t cpuSpeedStr [] = "  SysClock=";
const uint8_t debugStr [] = "   BackOP_O";
const uint8_t pin4Str [] = "   Pin4:";
const uint8_t nrstStr [] = "NRST ";
const uint8_t pa1Str [] = "PA1 ";
const uint8_t baudStr [] = "BaudRate=";
const uint8_t pendStr[] = "(pending)";
const uint8_t spiStr [] = "   SPI=";
const uint8_t freqStr [] = "FREQ=";
const uint8_t pwm1Str [] = "Hz   PWM1=";
const uint8_t pwm2Str [] = ",   PWM2=";
const uint8_t offStr [] = "OFF";
const uint8_t motStr [] = "    MOT=";
const uint8_t tstepStr [] = "Step=";
const uint8_t nstepsStr [] = "ms NSTP=";
const uint8_t dcmotStr [] = "DC/PWM1";


	sendTB('D');
	sendTB('a');
	sendTB('t');
	sendTB('a');
	sendString(stackStr);
	sendX32((uint32_t)(gfDataStack + 3));
	sendTB('R');
	sendTB('e');
	sendTB('t');
	sendTB('u');
	sendTB('r');
	sendTB('n');
	sendString(stackStr);
	sendX32((uint32_t)gfReturnStack);
	sendString(threadStr);
	sendX32((uint32_t)gfTCode);
	sendTB('U');
	sendTB('s');
	sendTB('e');
	sendTB('r');
	sendTB('=');
	sendX32((uint32_t)gfUserHeader);
	sendTB(CRR);

	if (gProject == 0) sendString(gfEmptyStr);
	else  {
		sendString(gfProjectStr);
		sendTB(gProject + 0x30);
	}
	sendString(autoStr);
	if (gfAuto == 0  )  {
		sendTB('-');
		sendTB('-');
	}
	else  {	
		for (k = 0;k < 8;k++)  {
			if (gfUserHeader[gfAuto-1].name[k] == 0)  break;
			sendTB(gfUserHeader[gfAuto-1].name[k]);
		}
	}

	sendString(debugStr);
	if (gfBackProcess == 0)  {
		sendTB('F');
		sendTB('F');
	}
	else sendTB('N');
	sendString(pin4Str);
	if (FLASH->OPTR & FLASH_OPTR_NRST_MODE_0) sendString(nrstStr);
	else sendString(pa1Str);
//	sendX32(FLASH->OPTR);
//	sendX32(SYSCFG->CFGR3);
	sendTB(CRR);

	sendString(baudStr);
	sendNS32(gBaud);
	if (gBaud == 11)  {
		sendTB('5');
		sendTB('2');
	}
	if (gBaud == 19)  sendTB('2');
	if (gBaud == 38)  sendTB('4');
	if (gBaud == 57)  sendTB('6');
	sendTB('0');
	sendTB('0');
	if (gBaudSet == 0) sendString(pendStr);
	sendString(spiStr);
	sendNS32(gfSpiMode);
	sendTB(',');
	if (gfTempNumBase==16) {
		send0X32(gfSpiBits);
		sendTB(',');
		send0X32(gfSpiClock);
	}
	else  {
		sendNS32(gfSpiBits);
		sendTB(',');
		sendNS32(gfSpiClock);
	}
		sendTB(CRR);
	sendString(freqStr);

	if (gfFreq <= 0) sendNS32(0);
	else  {
#ifdef CLOCK_48M
		k = gfFreq * 2000;
		q = 48000000 /(k);
		r = (48000000 - (q * k)) * 10;
#else
		k = gfFreq * 200;
		q = 3000000 /(k);
		r = (3000000 - (q * k)) * 10;
#endif
		sendNS32(q);
		sendTB('.');
		q = r /	k;
		r = (r - (q	* k)) *	10;
		sendNS32(q);
		q = r /	k;
		r = (r - (q	* k)) *	10;
		k = r /	k;
		if (k >= 5) q++;
			sendNS32(q);
	}
	sendString(pwm1Str);
	if (gfPWM1 < 0) sendString(offStr);
	else {
		if (gfTempNumBase==16) send0X32(gfPWM1);
		else  sendNS32(gfPWM1);
	}
	sendString(motStr);
	if (gfUserHeader[4].toCompile == -1)  {
		if (GPIOA->ODR & 2) sendTB('-');	//PA1 High = CCW
		sendString(dcmotStr);
	}
	else  {
		sendString(tstepStr);
		sendNS32(gfStepSpeed);
		sendString(nstepsStr);
		sendNS32(gfUserHeader[4].toCompile);
	}

	sendString(pwm2Str);
	if (gfPWM2 < 0) sendString(offStr);
	else {
		if (gfTempNumBase==16) send0X32(gfPWM2);
		else  sendNS32(gfPWM2);
	}

	sendTB(CRR);

}


uint32_t checkProjectNum(uint32_t mode)  //	*********************************
// *************************************************************************
{
uint32_t n;
const uint8_t useStr[] = " use actual code as ";
const uint8_t newStr[] = " start new ";
const uint8_t startupStr[] = " ..w.SystemStart Proj.";
	n = *gfSP;
	gfSP--;
	if (mode == 1)  {
		if ((n > 2) && (n != 10) && (n != 12)) {
			if (n > 2) 		gfErrors |= ERR_NUMVAL;
			return 0xFFFFFFFF;
		}
		if (n == 0) sendString(useStr);
		else sendString(gfSaveStr);
	}
	else {
		if (n > 2) {
			gfErrors |= ERR_NUMVAL;
			return 0xFFFFFFFF;
		}
		if (n == 0) sendString(newStr);
		else sendString(gfLoadStr);
	}

	if (n == 0) sendString(gfEmptyStr);
	else  {
		sendString(gfProjectStr);
		if (n > 2) {
			sendTB('1');
			sendString(startupStr);
			sendTB(n + 38); // sends 0 or 2
		}
		else sendTB(n + 0x30);
	}
	if (conFirm() == 0)  return 13;
	return n;
}


uint32_t conFirm(void) // *************************************************
//save some code length
// *************************************************************************
{
uint32_t k;
const uint8_t confirmStr [] = " ?..confirm w.upperCase 'Y'";

	sendString(confirmStr);
	while ((k= pullTermRxBuf()) == 0);
	if ((k & 0x000000FF) != 'Y')  {
		sendString(gfCancelStr);
		return 0;
	}
	sendTB(CRR);
	return 1;
}


void op_SAVEFLASH(void)  //	************************************************
// save this project state in microcontroller internal flash
// *************************************************************************
{
uint32_t n;
const uint8_t emptyStr[] = " = empty with pre-existing code";

	n = checkProjectNum(1);
	if (n > 12) return;
	gProject = n;
	if (n == 0) {
		sendString(gfProjectStr);
		sendTB('0');
		sendString(emptyStr);
		sendTB(CRR);
	}
	else pushFlashProject();
}


void op_LOADFLASH(void)  //	************************************************
// reload previously saved project from microcontroller internal flash
// *************************************************************************
{
uint32_t n;

	n = checkProjectNum(0);
	if (n > 2) return;

	gProject = n;		//valid temporarily until next CForth start
	if (gProject == 0)  {
		emptyForth();
		op_QUEST ();
	}
	else  popFlashProject(gProject + 2);
}


void checkBackNum(void)  //	************************************************
//save some code length
// *************************************************************************
{
	if (gfBackNumValid)  {
		gfBackNumValid = 0;
		gfSP++;
		*gfSP = gfBackProcData;
	}
}

// ### END #################################################################
