// CForth C011(SO8) project **************************************************
//
// Author and (c)2025 Wolfgang Schemmert. The copyright claim is restricted
// to this file and the depending files "main.h","cforth.c","cforth.h",
// and "globaldefines.h"
// Other depending material is published, copyrighted and licensed
// by STMicroelectronics and Segger GmbH
//
// Contact: <www.midi-and-more.de>
// A detailled hardware construction and operation manual
// is provided at this website
//
// ***************************************************************************
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//	Licence: GPL-3
//
// ***************************************************************************

#include "stm32c011xx.h"
#include "globaldefines.h"
#include "main.h"

int main(void)   // ********************************************************
// **************************************************************************
{
uint32_t recv;
uint32_t* RegAddr;

	SetSysClock();				  //48MHz or 3.0MHz HSI clock

	RCC->IOPENR |= RCC_IOPENR_GPIOAEN;		// Enable Port A clock
	RCC->IOPENR |= RCC_IOPENR_GPIOBEN;		// Enable Port B clock
	RCC->IOPENR |= RCC_IOPENR_GPIOCEN;		// Enable Port B clock
	RCC->APBENR2 |= RCC_APBENR2_SYSCFGEN;	// Enable SYSCFG clock

	GPIOA->MODER = 0x00000000;		//primarily all input
	GPIOB->MODER = 0x00000000;		//primarily all input
	GPIOC->MODER = 0x00000000;		//primarily all input
//existing inputs are configured individually:
	GPIOA->PUPDR = 0x00000000;		//primarily all inputs HiZ
	GPIOB->PUPDR = 0x00000000;		//primarily all  inputs HiZ
	GPIOC->PUPDR = 0x00000000;		//primarily all inputs HiZ

//assign PA8 to PA11(9) and pin4 to GPIO PA1
	if ( FLASH->OPTR & FLASH_OPTR_NRST_MODE_0) {		//NRST active
		SYSCFG->CFGR3 = SYSCFG_CFGR3_PINMUX2_0;
	}
	else SYSCFG->CFGR3 = (SYSCFG_CFGR3_PINMUX2_0 | SYSCFG_CFGR3_PINMUX1_1) ;
//remap PA11 t PA9
	SYSCFG->CFGR1 |= SYSCFG_CFGR1_PA11_RMP;

	gLEDCOUNT = 0;
	gSYSCOUNT = 0;
	gDELAYCOUNT = 0;
	gTIMEOUT = 0;
	gBaud = DEFAULT_BAUD;	//115200
	gfNumBase = DEFAULT_NUMBASE;
	gfTempNumBase = DEFAULT_NUMBASE;
	gProject = 0;
	gRandom = 314159265;	//seed for xorshift32 algorithm
	gTxOff = 0;

	SysTick_Config(SystemCoreClock / 1000);	//adjust to 1ms
											//for any SystemCoreClock

//no independent Watchdog, else conflict with SLEEP

//init I/O state  *************************
	op_ABORT();					//reset default Forth system data
	popFlashProject(0);			//loads only gBaud, gfNumBase and gProject
	msDelay(10);
//init communication  ********************
	initUsart1();
	msDelay(10);

	sendTB(CRR);
	sendString(gTypeStr);
	sendTB(HARDWARE_VERSION + 0x30);
	sendTB('.');
	sendNS32(REVNUM);
	sendString(gCopyStr); 
	sendTB(CRR);

	popFlashProject(1);				//if (gProject==0) start empty project
									//if virgin or gProject==0 emptyForth();
	while(1) {
	quit();
	if (query() == 0) break;
	if (gfTibMax == 0) continue;
	eval();
	}
}

// ##########################################################################

// Misc	Low	Level Routines
//************************

void SetSysClock(void)   // *************************************************
// *************************************************************************
{
uint32_t StartUpCounter = 0, HSIStatus = 0;

	RCC->CFGR = 0;		//clear all HPRE options, select HSI(confirm default)
	 do  {
		HSIStatus = RCC->CR & RCC_CR_HSIRDY;
	 StartUpCounter++;
	} while((HSIStatus == 0) && (StartUpCounter <= 0x5000));

	
//PCLK and HCLK	undivied by default -_> peripherals with max.speed
	RCC->CFGR &= 0;		//confirm

	RCC->CCIPR = RCC_CCIPR_USART1SEL_1;	//HSIKER(16MHz) selected for USART1
										//SYSCLK is selected for ADC

//HSI clock dividers
	RCC->CR &= ~(RCC_CR_HSIDIV);//SYSCLK=48MHz, HSIKERCLK=16MHz(default)
#ifdef CLOCK_48M
//Enable Prefetch Buffer //and set Flash Latency
	FLASH->ACR = FLASH_ACR_PRFTEN | 0x00000001 ;	//=max waitstates
	SystemCoreClock = 48000000;
#else
	RCC->CR |= RCC_CR_HSIDIV_2;						//dvide by 16 = 3.0MHZ
//Enable Prefetch Buffer //and set Flash Latency
	FLASH->ACR = FLASH_ACR_PRFTEN;			//=min waitstates
	SystemCoreClock = 3000000;
#endif
}


void msDelay(uint32_t count)   // *******************************************
//blocking delay by 'count' milliseconds
// **************************************************************************
{
	if (count == 0) return;
	if (count > 60000) count = 60000;	//max 1 minute
	gDELAYCOUNT = count;
	while(gDELAYCOUNT)  {
		servSystem();
	}
}


void wait5us(void)  //	*****************************************************
// blocking delay ca 5us
// **************************************************************************
{
uint32_t n;
#ifdef CLOCK_48M
	for (n=0;n<9;n++) {
		servSystem();
	}
#else
	for (n=0;n<1;n++) {
		servSystem();
	}
#endif
}


void servSystem(void) // ***************************************************
//not very time critical but permanently repeated functions for system control
//exclusively called in pullTermRxBuf() and some long-waiting loops
//this way it is regularily served when idle waiting for external input
//slowed down by SysTick to 50 times/second
//actually, it is more or less a dummy here, but kept for debugging assistance
// ***************************************************************************
{
uint32_t recv;
	if (gSYSCOUNT && (gSYSCOUNT	< 0x80000000)) return;		
						  //countdown in SysTick_Handler()
	gSYSCOUNT = 50;	  //restart 20Hz service interval
}


void popFlashProject(uint32_t mode) // *************************************
//read project independent system parameters
// **************************************************************************
{
uint32_t * flashPtr, *sramPtr;
uint32_t k, startProject, m;

	m= 0;
	emptyForth(); // primarily init defaults - implies gProject = 0;

	if (mode == 4) goto CHECK2;

	flashPtr = (uint32_t *)(PAGE12);
	k = *flashPtr; // can't be=0xFFFFFFFF if project was saved in Flash
	if (k == 0xFFFFFFFF) {
		if (mode == 3) {
			sendString(gfProjectStr);
			sendTB('1');
			goto CHECKX;
		}
		else goto CHECK2;
	}
	flashPtr++;
//system parameters
	k = ~(*flashPtr);
	flashPtr++;
	gBaud = k & 0x0000007F;
	startProject = (k >> 8) & 0x00000003;
	if (mode == 3)  startProject = 1;
	gfNumBase = (k >> 16) & 0x0000001F;
	if (startProject < 2) {
		gProject = startProject;
		goto CHECKOUT;
	}
	else if (startProject == 2) {
		mode = 4;
CHECK2:
		flashPtr = (uint32_t *)(PAGE14);
		k = *flashPtr; // can't be=0xFFFFFFFF if project was saved in Flash
		if (k == 0xFFFFFFFF) {
			if (mode == 4) {
				sendString(gfProjectStr);
				sendTB('2');
CHECKX:
				sendString(gEmptyStr);
				sendTB(CRR);
				gProject = 0;
			}
			goto CHECKOUT;
		}
		flashPtr++;
//system parameters
		k = ~(*flashPtr);
		flashPtr++;
		gBaud = k & 0x0000007F;
		gProject = (k >> 8) & 0x00000003;
		gfNumBase = (k >> 16) & 0x0000001F;
	}
	else  return;		//precaution, should be impossible

CHECKOUT:
	if  (mode == 0) return;

	if (gProject)  {
		gfAuto = ~(*flashPtr);
		flashPtr++;
		gfSpiMode = ~(*flashPtr);
		flashPtr++;
		k = ~(*flashPtr);
		flashPtr++;
		gfSpiBits = (k >> 16) & 0x0000007F;
		gfSpiClock = k& 0x0000007F;
		gfFreq = ~(*flashPtr);
		flashPtr++;
		gfPWM1 = ~(*flashPtr);
		flashPtr++;
		gfPWM2 = ~(*flashPtr);
		flashPtr++;
		if (gfFreq) {
			subFREQ();
			subPWM1();
			subPWM2();
		}

// first load UserToken Headers for project
// loading is possible over page limits
		sramPtr = (uint32_t *)gfUserHeader;
		for (k = 0;k < (MAXUSER * 4);k++)  {	
		//4 words per gfUserHeader entry->384 words=1536 bytes=3/4page
			*sramPtr = ~(*flashPtr);
			sramPtr++;
			flashPtr++;
		}

//next load UserToken code for project
		if (gProject == 1)  flashPtr = (uint32_t *)(PAGE13);
		else  flashPtr = (uint32_t *)(PAGE15);
		sramPtr = (uint32_t *)gfTCode;
		for (k = 0;k < MAXCODE/4;k++)  {	//2048(0x800) bytes = full page
			*sramPtr = ~(*flashPtr);
			sramPtr++;
			flashPtr++;
		}
	}

//from here if(gProject != 0) or not:
	msDelay(10);
	op_ABORT();					//reset project Forth system data
	if (gfAuto)  {
//emergency brake to avoid AUTO	deadlock
		if ((GPIOA->IDR & 0x400) == 0) gfAuto = 0;	//if PA10 (RX input)is low
	}
	op_QUEST();
	if (gfAuto) {
	gfParsed = gfAuto -	1;	//-1 to distinguish	from empty AUTOEXE == 0
	interpretUserToken();
	}
}

uint32_t pushFlashProject(void)    //******************************************
// save project independent system parameters permanently
//  **************************************************************************
{
uint32_t *flashPtr, *sramPtr;
uint32_t k, offset, startProject;

	startProject = gProject;
	if (gProject == 10) {
		gProject = 1;
		startProject = 0;
	}
	if (gProject == 12) {
		gProject = 1;
		startProject = 2;
	}
	if ((gProject == 0) || (gProject > 2)) return 0;//precaution
													//don't save empty project

//unlock flash
#define KEY1    0x45670123		//mandatory unlock key defined by STM
#define KEY2    0xCDEF89AB

	while ((FLASH->SR &FLASH_SR_BSY1) !=0); //wait until no operation going on
	if ((FLASH->CR & FLASH_CR_LOCK) != 0) {
		FLASH->KEYR = KEY1;
		FLASH->KEYR = KEY2;
	}
	else return 0;

	offset = (gProject - 1) * 2;	//project no.0 is empty, not treated here
									//project no.1 starts with page#24
//erase Page12 + offset (=Page14)
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;  //wait until flash is not busy
	FLASH->CR = (((12+offset)<<3)|FLASH_CR_EOPIE|FLASH_CR_ERRIE|FLASH_CR_PER);
									 //EOPIE is set formally to get SR_EOP set
	FLASH->CR |= FLASH_CR_STRT;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
	FLASH->SR |= FLASH_SR_EOP;
	if (FLASH->SR & FLASH_SR_OPERR)  return 0;

FLASH->CR &= ~FLASH_CR_PER;

//this microcontroller allows exclusively doubleword-wide write in flash

	flashPtr = (uint32_t *)(PAGE12 +  offset * 0x800);
	FLASH->CR |= FLASH_CR_PG;
//first 32 bytes with system & peripheral config parameters
	*flashPtr = 0x00000000; //virgin flash test
	flashPtr++; 
	*flashPtr = ~((gfNumBase << 16) | (startProject << 8) | gBaud);
	flashPtr++;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
	FLASH->SR |= FLASH_SR_EOP;
	if (FLASH->SR & FLASH_SR_OPERR) return 0;

	 *flashPtr = ~gfAuto;
	flashPtr++;
	*flashPtr = ~(gfSpiMode);
	flashPtr++;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
	FLASH->SR |= FLASH_SR_EOP;
	if (FLASH->SR & FLASH_SR_OPERR)  return 0;

	*flashPtr = ~((gfSpiBits << 16 ) | gfSpiClock);
	flashPtr++;		
	*flashPtr = ~gfFreq;
	flashPtr++;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
	FLASH->SR |= FLASH_SR_EOP;
	if (FLASH->SR & FLASH_SR_OPERR) return 0;

	*flashPtr = ~((uint32_t)gfPWM1);
	flashPtr++;	
	*flashPtr = ~((uint32_t)gfPWM2);
	flashPtr++;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
	FLASH->SR |= FLASH_SR_EOP;
	if (FLASH->SR & FLASH_SR_OPERR)  return 0;

// following write gfUserHeader[] into (Page12 + offset*0x800 + 8)
// each= 4words/16 bytes/struct*96elements=384 words=1536bytes = 3/4 page
	sramPtr = (uint32_t *)&(gfUserHeader[0]);
	for (k = 0;k < MAXUSER*2; k++) {	//MAXUSER=96, 4 words per entry 	
	// this microcontroller allows exclusively doubleword-wide write in flash
	//first word
		*flashPtr = ~(*sramPtr); //~for better compatibility with empty SRAM
		flashPtr++;
		sramPtr++;
	//second word
		*flashPtr = ~(*sramPtr);
		flashPtr++;
		sramPtr++;
	//burn
		while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
			FLASH->SR |= FLASH_SR_EOP;
			if (FLASH->SR & FLASH_SR_OPERR)  return 0;
	}

	sendTB('.');
//terminate
	FLASH->CR &= ~FLASH_CR_PG;

	while ((FLASH->SR &FLASH_SR_BSY1) !=0); //wait until no operation going on
	if ((FLASH->CR & FLASH_CR_LOCK) != 0) {
		FLASH->KEYR = KEY1;
		FLASH->KEYR = KEY2;
	}

//erase Page13 + offset (=Page15)
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;  //wait until flash is not busy
	FLASH->CR = (((13+offset)<<3)|FLASH_CR_EOPIE|FLASH_CR_ERRIE|FLASH_CR_PER);
									//EOPIE is set formally to get SR_EOP set
	FLASH->CR |= FLASH_CR_STRT;
	while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
	FLASH->SR |= FLASH_SR_EOP;
	if (FLASH->SR & FLASH_SR_OPERR)  return 0;

	FLASH->CR &= ~FLASH_CR_PER;

//write user code[] into Page13 + offset
	flashPtr = (uint32_t *)(PAGE13 + offset*0x800);
	FLASH->CR |= FLASH_CR_PG;
// this microcontroller allows exclusively doubleword-wide write in flash
	sramPtr = (uint32_t *)gfTCode;
	for (k = 0; k < MAXCODE/4; k += 2)  {	//MAXCODE=0x800=204bytes= full page
	//first word
		*flashPtr = ~(*sramPtr); //~for better compatibility with empty SRAM
		flashPtr++;
		sramPtr++;
	//second word
		*flashPtr = ~(*sramPtr);
		flashPtr++;
		sramPtr++;
//burn
		while ((FLASH->SR & FLASH_SR_BSY1) != 0) ;
		FLASH->SR |= FLASH_SR_EOP;
		if (FLASH->SR & FLASH_SR_OPERR)  return 0;
	}
//terminate
	FLASH->CR &= ~FLASH_CR_PG;
	sendTB('.');

//lock flash
	FLASH->CR |= FLASH_CR_LOCK;
	msDelay(10);
	return 1;
}


//###########################################################################

//USART	Support
//*************
void initUsart1(void) // ****************************************************
// USART1-TxD is PortA,9
// USART1-RxD is PortB,7
// USART1 clock is always derived from SYSCLOCK= 48	MHz
// **************************************************************************
{
uint32_t baudset;

	RCC->APBENR2 |= RCC_APBENR2_USART1EN ;	// USART1 clock enable

//repeat as precaution:
//assign PA8 to PA11
//	SYSCFG->CFGR3 |= SYSCFG_CFGR3_PINMUX2_0;
//remap PA11 to PA9
//	SYSCFG->CFGR1 |= SYSCFG_CFGR1_PA11_RMP;

//set PA9 in alternate function mode
	GPIOA->MODER &= ~ (GPIO_MODER_MODE9_0);
	GPIOA->MODER |= GPIO_MODER_MODE9_1;
//USART1 is AF1	on PortA
	GPIOA->AFR[1] &= 0xFFFFFF0F;	//PA9 is TX
	GPIOA->AFR[1] |= 0x00000010;
//set PB7 in alternate function mode
	GPIOB->MODER &= ~ (GPIO_MODER_MODE7_0);
	GPIOB->MODER |= GPIO_MODER_MODE7_1;
//USART1 is AF0 on PortB7,
	GPIOB->AFR[0] &= 0x0FFFFFFF;	//PB7 is RX

//	USART1->CR1 &= ~(USART_CR1_UE);	//disable USART - else no Baudrate change
		switch(gBaud) {
		case 96:
			baudset = 0x683;	//9600Baud: 104.17 at 16 MHz HSIKER clk
			break;
		case 19:
			baudset = 0x0341;	//19200Baud: 52.08 at 16 MHz HSIKER clk
			break;
		case 38:
			baudset = 0x01A1;	//38400Baud: 26.04 at 16 MHz HSIKER clk
			break;
		case 57:
			baudset = 0x0116;	//57600Baud: 17.36 at 16 MHz HSIKER clk
			break;
		case 11:
		default:
			baudset = 0x008B;	//default (virgin flash)
								//115200Baud: 8.68 at 16 MHz HSIKER clk
		}

	USART1->BRR = baudset;
	USART1->CR1 = USART_CR1_RXNEIE_RXFNEIE | USART_CR1_TE | USART_CR1_RE;
											//8Bit, no Parity, RX Int
	USART1->CR2 = USART_CR2_TXINV;			//1 Stopbit, TX	level inverted
	USART1->CR3 = 0;	
	USART1->CR1	|= USART_CR1_UE;	//enable USART
//default:no Handshake
	gUSART_RXHEAD = gUSART_RXBUF;
	gUSART_RXTAIL = gUSART_RXBUF;
	gBaudSet = 1;
	NVIC_EnableIRQ(USART1_IRQn);
	msDelay(2);
}


void sendTB(uint8_t data) // *************************************************
//send 1 raw byte
// **************************************************************************
{
	if (gTxOff) return;
	while(!(USART1->ISR & USART_ISR_TXE_TXFNF));//=bit7 =TXE in no_FIFO mode
	USART1->TDR = data;
}


void sendNibble(uint8_t data) // ********************************************
// **************************************************************************
{
uint8_t asciiNibble;

	asciiNibble = data+0x30;
	if (asciiNibble>0x39) asciiNibble+= 7;
	sendTB(asciiNibble);
}


void sendString(const uint8_t *	str) //	*************************************
// **************************************************************************
{
 uint8_t byte;

	while ((byte = *str++) != 0)  sendTB(byte);
}


void sendX8(uint32_t data) // *********************************************
// send bits 0...7 in hex format (without leading 0x)
// **************************************************************************
{
	uint8_t temp;

	temp= (data & 0xF0) >> 4;
	temp = temp +0x30;
	if (temp > 0x39) temp+= 7;
	sendTB(temp);
	temp= data & 0x0F;
	temp = temp	+ 0x30;
	if (temp > 0x39) temp += 7;
	sendTB(temp);
	sendTB(0x20);
}


void sendX32(uint32_t data) // **********************************************
// send 1 word (32bit) in hex format (without leading 0x)
// Leading zeroes are suppressed
// **************************************************************************
{
uint8_t nibble;
	nibble = (uint8_t) ((data>>28) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) ((data>>24) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) ((data>>20) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) ((data>>16) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) ((data>>12) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) ((data>>8) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) ((data>>4) & 0x0000000F);
	sendNibble(nibble);
	nibble = (uint8_t) (data & 0x0000000F);
	sendNibble(nibble);
	sendTB(0x20);
}


void sendNX32(uint32_t data) //	**********************************************
// send 1 word (32bit) in hex format (without leading 0x)
// **************************************************************************
{
uint8_t nibble, noLead;
	noLead = 0;
	nibble = (uint8_t) ((data>>28) & 0x0000000F);
	if (nibble)  {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) ((data>>24) & 0x0000000F);
	if (nibble || noLead) {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) ((data>>20) & 0x0000000F);
	if (nibble || noLead) {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) ((data>>16) & 0x0000000F);
	 if (nibble || noLead) {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) ((data>>12) & 0x0000000F);
	if (nibble || noLead) {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) ((data>>8) & 0x0000000F);
	if (nibble || noLead) {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) ((data>>4) & 0x0000000F);
	if (nibble || noLead) {
	sendNibble(nibble);
	noLead = 1;
	}
	nibble = (uint8_t) (data & 0x0000000F);
	sendNibble(nibble);
}


void send0X32(uint32_t data) //	**********************************************
// send 1 word (32bit) in hex format WITH leading 0x.Leading zeroes suppressed
// **************************************************************************
{
//	if (data >= 10)  {			//modified 11Jan24
		sendTB('0');
		sendTB('x');
//	}
	sendNX32(data);
}


void sendS32(uint32_t data) // *******************************************
//send signed decimal number string. Leading zeroes are suppressed
//trailing SPACE
//up to  +/- 2.147.483.647 Billions
// **************************************************************************
{
	sendNS32(data);
	sendTB(0x20);
}


 void sendLS32(uint32_t data) // ********************************************
//send signed decimal number string. Leading zeroes are suppressed
//leading SPACE
//up to  +/- 2.147.483.647 Billions
// **************************************************************************
{
	sendTB(0x20);
	sendNS32(data);
}


void sendNS32(uint32_t data) //	*******************************************
//send signed decimal number string. Leading zeroes are suppressed
//no leading and trailing SPACE
//up to  +/- 2.147.483.647 Billions
// **************************************************************************
{
uint32_t quotient;

	if (data >= 0x80000000) {
	sendTB('-');
	data = ~data + 1 ;
	}
	if (data < 10) goto ONES;
	if (data < 100) goto TENS;
	if (data < 1000) goto HUNDREDS;
	if (data < 10000) goto THOUSANDS;
	if (data < 100000) goto TH10;
	if (data < 1000000) goto TH100;
	if (data < 10000000) goto MILLIONS;
	if (data < 100000000) goto MIL10;
	if (data < 1000000000) goto MIL100;
	quotient = data/1000000000;
	data = data - quotient*1000000000;
	sendTB((uint8_t)quotient+ 0x30);
MIL100:
	quotient = data/100000000;
	data = data - quotient*100000000;
	sendTB((uint8_t)quotient + 0x30);
MIL10:
	quotient = data/10000000;
	data = data - quotient*10000000;
	sendTB((uint8_t)quotient + 0x30);
MILLIONS:
	quotient = data/1000000;
	data = data - quotient*1000000;
	sendTB((uint8_t)quotient + 0x30);
TH100:
	quotient = data/100000;
	data = data - quotient*100000;
	sendTB((uint8_t)quotient+ 0x30);
TH10:
	quotient = data/10000;
	data = data - quotient*10000;
	sendTB((uint8_t)quotient + 0x30);
THOUSANDS:
	quotient = data/1000;
	data = data - quotient*1000;
	sendTB((uint8_t)quotient + 0x30);
HUNDREDS:
	quotient = data/100;
	data = data - quotient*100;
	sendTB((uint8_t)quotient+ 0x30);
TENS:
	quotient = data/10;
	data = data - quotient*10;
	sendTB((uint8_t)quotient + 0x30);
ONES:
	sendTB((uint8_t)data + 0x30) ;
}


uint32_t pullTermRxBuf(void) //	********************************************
// pull 1 byte asynchronously from USART reveive buffer
// check first if something received. If yes then set bit 15. If not return 0
// less overhead and more safe than use of a separate "buffer filled" flag
// **************************************************************************
{
uint32_t recv;

	servSystem();		//serve LED

	if (gUSART_RXTAIL != gUSART_RXHEAD)  {
		recv= (uint32_t)(*gUSART_RXTAIL);
		recv |= 0x00008000;
		gUSART_RXTAIL++; 
		if (gUSART_RXTAIL==gUSART_RXBUFTOP)  gUSART_RXTAIL=gUSART_RXBUF;
		return recv;
	}
	else return 0;
}


// Interrupt Handlers
//*******************

void SysTick_Handler(void) // ***********************************************
// ***************************************************************************
{
uint32_t recv;

	if (gSYSCOUNT) gSYSCOUNT--;		//controls servSystem()
	if (gDELAYCOUNT) gDELAYCOUNT--;
	if (gTIMEOUT) gTIMEOUT--;		//to be used in any subroutine
	if (gfMSCOUNT) gfMSCOUNT --;		//used by MS, DS
	if (gfTIXCOUNT) gfTIXCOUNT --;		//used by  TIX
	if (gfSTEPCOUNT	> 0) gfSTEPCOUNT--;	//stepper timing
	gRandom ^= gRandom << 13;		//xorshift32 algorithm/white noise
	gRandom	^= gRandom >> 17;
	gRandom	^= gRandom << 5;
}

void USART1_IRQHandler(void) // *****************************************
// *************************************************************************
{
uint32_t test;

  if (USART1->ISR & USART_ISR_RXNE_RXFNE)  {//=bit5 = RXNE in no_FIFO mode 	

	test = USART1->RDR & 0x000000FF;
	*gUSART_RXHEAD = (uint8_t)test;

	  if (test == CTRL_C)  {
	  gfTermInput = CTRL_C;
	  return;
	  }
	  if (test != CTRL_B)  {
	  *gUSART_RXHEAD = (uint8_t)test;
	  gUSART_RXHEAD++;
	  if (gUSART_RXHEAD==gUSART_RXBUFTOP)  gUSART_RXHEAD=gUSART_RXBUF;
	  }
	  else op_BAKOP();
  }
}

// #########################################################################
