// Forth specific routines of of the CForth LowPower project *****************
//
// Author and (c)2020-25 Wolfgang Schemmert. The copyright claim is restricted
// to this file and the depending files "main.c", "main.h", "cforth.h",
// and "globaldefines.h"
// Other depending material is published, copyrighted and licensed
// by STMicroelectronics and Segger GmbH
//
// Contact: <www.midi-and-more.de>
// A detailled hardware construction and operation manual
// is provided at this website
//
// ****************************************************************************
//      This program is free software: you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation, either version 3 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//      Licence: GPL-3
//
// ****************************************************************************

#include "stm32f0xx.h"
#include "globaldefines.h"
#include "cforth.h"


void quit(void)   // **************************************************
// ********************************************************************
{
uint32_t k;
//const uint8_t ErrorStr [] = "Error:";
const uint8_t ColStr [] = " (col ";
const uint8_t NotFoundStr [] = "_not found";
const uint8_t SyntaxStr [] = "_Syntax Err";
const uint8_t StackStr [] = "_Stack Err";
const uint8_t CompoStr [] = "_Compile only";
const uint8_t CodeAccessStr [] = "_Invalid Code access";
const uint8_t InvalContextStr [] = "_out of Context";
const uint8_t RunOnlyStr [] = "_Runtime only";
const uint8_t CancelStr [] = "_Thread cancelled";
const uint8_t RangeStr [] = "_Number out of Range";
const uint8_t Div0Str [] = "_Div.by Zero";
const uint8_t CompilingStr [] = "  compiling..";

    if (gfSP >= gfDataStack + MAXDATASTACK - 4) gfErrors |= ERR_STACK;
    if (gfSP < gfDataStack + 3) gfErrors |= ERR_STACK;

	if (gfErrors != 0)  {
		if (gfCompiState) {
           for (k=0; k<8; k++) {
               gfUserHeader[gfNextNameCompi].name[k] = 0;
           }
               gfUserHeader[gfNextNameCompi].toCompile = 0;
               gfUserHeader[gfNextNameCompi].attributes = 0;
		}

		gfCompiState = 0;		//terminate compilation in any case

		if (gfTokenStart)  {
            for (k = 0;k < gfTokenStart;k++) sendTB(SPACE);
            sendTB('^');
            sendTB(CRR);
        }
        sendString(gfErrorStr);
		sendTB(':');
		if ((gfErrors & ERR_STRINGTOKEN) != 0)  sendString(NotFoundStr);
		if ((gfErrors & ERR_SYNTAX) != 0)  sendString(SyntaxStr);
		if ((gfErrors & ERR_STACK) != 0)  sendString(StackStr);
		if ((gfErrors & ERR_COMPONLY) != 0)  sendString(CompoStr);
		if ((gfErrors & ERR_USERCODE) != 0)  sendString(CodeAccessStr);
		if ((gfErrors & ERR_CONTEXT) != 0)  sendString(InvalContextStr);
        if ((gfErrors & ERR_CANCEL) != 0)  sendString(CancelStr);
        if ((gfErrors & ERR_RUNTOKEN) != 0)  sendString(RunOnlyStr);
		if ((gfErrors & ERR_NUMVAL) != 0)  sendString(RangeStr);
		if ((gfErrors & ERR_DIVZERO) != 0)  sendString(Div0Str);
        sendString(ColStr);
		sendNS32(gfTokenStart);
        sendTB(0x29);  //')'
        op_ABORT();			//resets gfErrors, too
	}
	else  {
        if (gfCompiState) {              //compiling
            sendString(CompilingStr);
		}
		else if (gfComment != 1) {  //interpreting and no full line comment
            sendTB(SPACE);
            sendTB('o');
            sendTB('k');
			if (gfTempNumBase == 16)  sendTB('H');
		}
	}
	sendTB(CRR);
    gfBackNumValid = 0;
}


uint32_t query(void)   // **************************************************
// *************************************************************************
{
uint32_t tibstate;
uint32_t recvword;
const uint8_t RestartStr [] ="COLD RESET? ";

    gfTibPos = 0;
    gfTibMax = 0;
    gfComment = 0;          //reset and start every line not in comment state
	
	do {
#ifndef USARTIO
//serve Background process
		if (gfStepSpeed)  {
			if ((gfUserHeader[4].toCompile > 0) ||
								(gfUserHeader[4].toCompile == 0xFFFFFFFE))  {
				if (gfSTEPCOUNT == 0) subMOT();
			}
		}
#endif
		tibstate = subQuery();
	} while(tibstate == 0);
	return 1;
}


uint32_t subQuery(void)   // ************************************************
// **************************************************************************
{
uint32_t recvword;
uint8_t recvbyte;

	recvword = pullTermRxBuf();
	if (recvword == 0)  return 0;
	recvbyte = (uint8_t)(recvword & 0x000000FF);
	if (gfBackProcess && gfExecUser)  {
		gfTermChar = recvbyte;
		gfTermCheck = 1;
	}
	if (recvbyte == LFF)  return 0;	//ignore LFF
	if (recvbyte <= CTRL_B)  return 0;	//wrong TIB format,ignore and go on

//if not executing User Token, convert TAB -> SPACE
	if (gfExecUser == 0)  {
		if (recvbyte == TAB)  recvbyte = SPACE;
		if ((gfTibPos >= 64) && (recvbyte == SPACE))  recvbyte = CRR;
								//force line break if line is too long
	}

//block loading comments:
	if ((recvbyte == 0x28) && (gfComment == 0)) {	//0x28 = '('
		sendTB(0x28);
		while ((recvword = pullTermRxBuf()) == 0);
		recvbyte = (uint8_t)(recvword & 0x000000FF);
		if (recvbyte == SPACE) {					// ( ..style comment
			gfComment = gfTibPos + 1;
			sendTB(BKSP);
			sendTB(SPACE);
			sendTB(BKSP);
		}
		else  {
			gfTib[gfTibPos] = 0x28;
			gfTibPos++;
			gfTib[gfTibPos] = recvbyte;
			gfTibPos++;
			sendTB(recvbyte);
		}
		return 0;
	}

	if ((recvbyte == 0x2F) && (gfComment == 0)) {	//0x2F = '/'
		sendTB(0x2F);
		while ((recvword = pullTermRxBuf()) == 0);
		recvbyte = (uint8_t)(recvword & 0x000000FF);
		if (recvbyte == 0x2F) {						// // style comment
			gfComment = gfTibPos + 1;
			sendTB(BKSP);
			sendTB(SPACE);
			sendTB(BKSP);
		}
		else  {
			gfTib[gfTibPos] = 0x2F;
			gfTibPos++;
			gfTib[gfTibPos] = recvbyte;
			gfTibPos++;
			sendTB(recvbyte);
		}
		return 0;
	}

	if (recvbyte == ESC)  {
		if (gfExecUser)  {				//-->termination of User Thread
			sendTB(CRR);
			gfTib[gfTibPos] = SPACE;
			gfTib[gfTibPos + 1] = 0;
			gfTibMax = gfTibPos;
			return 2;
		}
		else return 0;
	}

	if  (recvbyte == TAB)  return 3;
	if  (recvbyte == CTRL_R)  return 4;

	if ((recvbyte == BKSP) && (gfComment == 0)) {
		if (gfTibPos > 0)  {
			gfTibPos--;
			gfTib[gfTibPos]=SPACE;
			sendTB(BKSP);
			sendTB(SPACE);
			sendTB(BKSP);
		}
		return 0;
	}
//avoid line feed after empty comment lines:
   if (recvbyte == CRR)  {
		if (gfTibPos && (gfComment != 1)) sendTB(CRR);
		gfTib[gfTibPos] = SPACE;		//final token terminator for "eval"
		gfTib[gfTibPos + 1] = 0;
		gfTibMax = gfTibPos;
		return 1;
	}
//finally:		
	if (gfComment == 0)  {
		gfTib[gfTibPos] = (uint8_t)recvbyte;
		sendTB((uint8_t)recvbyte);
		gfTibPos++;
	}
	return 0;
}


void eval(void)   // ******************************************************
//evaluate "normal" terminal input
// ************************************************************************
{
	gfExecUser = 0;
	gfTokenStart = 0;
	gfTibPos = 0;

	while (gfTibPos < gfTibMax)  {
		scanNext();
		if (gfTokenstr[0] == 0) continue;	//ignore CTRL_B,TAB,LFF,ESC 

//return of interpretNumber():1:number, 0:look for token
		if (interpretNumber()) {			//valid number parsed
			if (gfCompiState == 0)  {		// interpreting
				gfSP ++;
				*gfSP = gfTokenNum;
			}
			else  {
				if ((uint32_t)gfTokenNum < 256)  {	
							//bugfix16Jun22 (uint32_t), gfTokenNum is int32_t
					compileTCode(IDX_DOLIT8B);
					compileTCode(gfTokenNum);
                }
				else  {
					compileTCode(IDX_DOLIT);
					compileTCode(gfTokenNum>>24); 
											//only bits 0..7 are compiled!
					compileTCode(gfTokenNum>>16);
					compileTCode(gfTokenNum>>8);
					compileTCode(gfTokenNum);
                }
			}
			gfTokenStart = gfTibPos;
			continue;
		}
// could not parse as number, try to interpret as token
		else if (findUserToken()) {	//User Token found, index in gfParsed
			if(interpretUserToken())  return;	//returns 0 if ok !!
			gfTokenStart = gfTibPos;
			continue;
		}
		else  {
			if (findKernelToken() == 0) {	// Kernel Tokenstring not found
			gfErrors |= ERR_STRINGTOKEN;
			gfCodeIndex = gfCodeStartIndex;
			gfTokenStart = gfTibPos;
			return;
			}
//Kernel Tokenstring found;
			interpretKernelToken(); //Kernel token found, index in gfParsed
			if (gfErrors) return;
			gfTokenStart = gfTibPos;
			continue;
		}
	}
}


void backProcEval(void)   // **************************************************
//while User Token is executing, evaluate "background process" terminal input
//in case of errors, send a message but don't set an error flag
// ************************************************************************
{
uint32_t k;
const uint8_t invalidStr [] = " <-not Kernel OP nor number";
const uint8_t noBackStr [] = " <-invalid for Back-Ctrl";
const uint8_t runErrStr [] = " <-Runtime ";

    for (k=82;k<gfTibMax;k++) sendTB(gfTib[k]); //for info, send parsed string

	gfTibPos = 82;
	gfComment = 0;
	gfTokenStart = gfTibPos;

	while (gfTibPos < gfTibMax)  {
		scanNext();
		if (gfTokenstr[0] == 0) continue;	//ignore CTRL_B,TAB,LFF,ESC 

		if (interpretNumber()) {	//valid number parsed, here only interpreting
			gfBackProcData = gfTokenNum;
			if (gfExecUser) gfBackNumValid = 1;
			gfTokenStart = gfTibPos;
			continue;
		}
		else  {
//result == 0: could not parse as number, try to interpret as token
            if (findKernelToken() == 0) {	// Kernel Tokenstring not found
                sendString(invalidStr);
                goto ERROR_BREAK;
            }
            //else Kernel Tokenstring found;
            if (gfKernelAttributes[gfParsed] & A_BAK) { //only if BAK attribute
                interpretKernelToken(); //Kernel token found, index in gfParsed
                if (gfErrors)  {
					gfErrors = 0;		//don't cancel background process
                    sendString(runErrStr);
					sendString(gfErrorStr);
					goto ERROR_BREAK;
                }
            }
            else  {
                sendString(noBackStr);
                goto ERROR_BREAK;
            }
            gfTokenStart = gfTibPos;
            continue;
		}
    }
ERROR_BREAK:
    sendTB(CRR);
    gfTibMax = 82;
    gfTibPos = 82;
    gfComment = 0;
}


void scanNext(void)   // **************************************************
// updates gfTokenstr[] with the next name string in TIB
// ************************************************************************
{
uint32_t k, tib;

	while (gfTib[gfTokenStart] == SPACE) gfTokenStart++;
								//eliminate leading SPACEs

	if (gfTib[gfTokenStart] == 0)  {	//is the case if CTRL_B,TAB,LFF,ESC
		gfTokenstr[0] = 0;
		gfTibPos = gfTokenStart;
		return;
	}

	k = 0;
	while ((tib = gfTib[gfTokenStart+k]) != SPACE)  {
								//search trailing SPACE(s) after scanned token
		gfTokenstr[k] = tib;
		k++;
		if (k > 14) break;	//limit length of string Bugfix 26May2022
						//(max.length of number. Code names are limited later
	}
	gfTokenstr[k] = 0;
	gfTibPos = gfTokenStart + k;
}


uint32_t interpretNumber(void)   // *****************************************
// hex numbers can be entered without leading 0X, gfTempNumBase is criterium
// optionally, when entered with leading 0X, Hex base is assumed temporarily
// leading '-'(after 0x if present) is accepted(good for small negative numbers)
// to simplify parameter transfer, succesfully parsed number is in "gfTokenNum"
//return:
// 1 = valid number
// 0 = no valid number, try to interpret as command token
// *************************************************************************
{
uint32_t numLen;

    numLen = 0;
    while (gfTokenstr[numLen] != 0) numLen++;

	if ((gfTokenstr[0] == '-') && (numLen == 1))  return 0;  //"-" operator	

	if ((gfTokenstr[0] == '0') &&
                    ((gfTokenstr[1] == 'X') || (gfTokenstr[1] == 'x'))) {
        if (gfTokenstr[2] == '-')  {
			//bugfix 11June22:wrong handling length of big negative numbers
			if (numLen > 11)  return 0;
		}
        else if (numLen > 10)  return 0;	//try to parse as command token
		if (string2Int(&gfTokenstr[2], 16))  return 1;
                    // success: interpreted as number, result in "gfTokenNum"
	}
	else if (gfTokenstr[0] == '%') {		//new feature 19Jan24
        if (gfTokenstr[1] == '-')  {
			if (numLen > 12)  return 0;
//			if (numLen > 12)  return 2;		//too long, not accepted as name
		}
        else if (numLen > 11)  return 0;	//try to parse as command token
//        else if (numLen > 11)  return 2;	//too long, not accepted as name
		if (string2Int(&gfTokenstr[1], 10))  return 1;
                    // success: interpreted as number, result in "gfTokenNum"
	}
	else {
        if (gfTempNumBase == 16)  {
			if (gfTokenstr[0] == '-')  {
				if (numLen > 9)  return 0;
			}
            else if (numLen > 8)  return 0;	//try to parse as command token
        }
        else  {
			if (gfTokenstr[0] == '-')  {
				if (numLen > 11)  return 0;
			}
			else if (numLen > 10)  return 0;    //name is too long
		}
        if (string2Int(&gfTokenstr[0], gfTempNumBase))  return 1;
                    // success: interpreted as number, result in "gfTokenNum"
    }
	return 0; //string doesnt represent a number:try to parse as command token
}


uint32_t string2Int(uint8_t* numStr, uint32_t numBase)   // ******************
//return: 1= valid number, 0=no valid number, try to interpret as command token
// ***************************************************************************
{
uint32_t n, length;
uint8_t numString[16];
uint32_t digit;

    n = 0;
    for (length = 0; length < 16;length++) {
        numString[length] = numStr[length];
    //transform into capital letters:
		if ((numString[length] >= 'a') && (numString[length] <= 'z'))
                                                    numString[length] -= 0x20;
        if (numString[length] == 0)  break;
    }
    numString[15] = 0;              //in any case, doesn't harm if length<15
    if ((numString[0] == '-') && (length == 1))  return 0;  //"-" operator

    gfTokenNum = 0;
    n = 0;
    if (numString[0] == '-') n = 1;       //bugfix 20.06.20
    while (n < length)  {
        digit = checkDigit(numString[n], numBase);
        if (digit == 0xFFFFFFFF) {
            return 0; //string doesn't represent a number
        }
        gfTokenNum = (int32_t)((gfTokenNum * numBase) + digit);
        n++;
    }
    if  (numString[0] == '-') {
        gfTokenNum = -gfTokenNum;
    }
    return 1;        				//string is parsed as valid number
}


uint32_t checkDigit(uint8_t digit, uint32_t numBase)   // *******************
// *************************************************************************
{
    if (digit < 0x30) return 0xFFFFFFFF;
    digit -= 0x30;
    if (digit < 10) return (int32_t)digit;
    if (numBase == 10) return 0xFFFFFFFF;
    else {
        if (digit < 0x11) return 0xFFFFFFFF; //error if digit < 0x41 (='A')
        if (digit > 0x16) return 0xFFFFFFFF; //error if digit > 0x46 (='F')
        digit -= 7;                         //range 0....0x0F
    }
    return digit;
}


uint32_t findKernelToken(void)   // ****************************************
// *************************************************************************
{
uint32_t k, n;
uint8_t kernelChar, refChar;

    k = 0;					//index 0 is op_ERR
	while (k < KERNELSIZE)  {
        for (n=0; n < 8;n++) {
            kernelChar = gfKernelNames[k][n];
            refChar = gfTokenstr[n];
//transform to capital letters:
            if ((kernelChar >= 'a')&&(kernelChar <= 'z')) kernelChar -= 0x20;
            if ((refChar >= 'a')&&(refChar <= 'z')) refChar -= 0x20;
            if (kernelChar == SPACE) kernelChar = 0;
            if (kernelChar != refChar) break;
            if (refChar != 0) continue;
            gfParsed = k;            //="gfKernelNames[]"index of found token
                    //=same index in "gfKernelOps[]" and "gfKernelAttibutes[]"
            return 1;
        }
		k++;
	}
	return 0;
}


uint32_t findUserToken(void)   // ******************************************
// *************************************************************************
{
int32_t k;
uint32_t n;
uint8_t refChar, userChar;

    k = 0;

	while (gfUserHeader[k].name[0] != 0)	k++;
	k--;            					//k is index of last compiled USER Token

	while (k != -1)  {			//start search from the last compiled
		for (n=0; n < 8;n++) {
            userChar = gfUserHeader[k].name[n];
            refChar = gfTokenstr[n];
//transform into capital letters:
            if ((refChar >= 'a')&&(refChar <= 'z')) refChar -= 0x20;
            if (userChar != refChar)  break;
            if (userChar == 0) {
                gfParsed =  k;          //="gfUserHeader[]" index of found token
                return 1;
            }
		}
		k--;
	}
	return 0;
}


void interpretKernelToken(void)	// *****************************************
// *************************************************************************
{
    if (gfKernelAttributes[gfParsed] & A_RUN) {
        gfErrors |= ERR_RUNTOKEN;
        return;
    }

	if(gfCompiState == 0) {  //interpreting, executing threaded code
		if (gfKernelAttributes[gfParsed] & A_CPO) {
			gfErrors |= ERR_COMPONLY;
			return;
		}
		gfKernelOps[gfParsed]();		//execute normally
	}
	else {								// compiling is active
		if (gfKernelAttributes[gfParsed] & A_IME) {
			gfKernelOps[gfParsed]();	//don't compile, execute immediately
		}
		else  compileTCode(gfParsed);	//compile this token
	}
}


uint32_t interpretUserToken(void)   // *************************************
//essential global parameter:gfParsed 
// unusual return value: 0= ok, else=Error
// *************************************************************************
{
uint32_t oldTibMax, oldTibPos;

	if(gfCompiState == 0) {	//interpreting, COMPONLY Option not accepted
		if ((gfUserHeader[gfParsed].attributes & A_VARCON) == 0) {
            gfIP = (uint8_t*)gfUserHeader[gfParsed].toCompile;
			gfExecUser = 1; 					 //modify query() behaviour
			oldTibMax = gfTibMax;
			oldTibPos = gfTibPos;
			gfTibMax = 82;		//intermediate TIB for background process
			gfTibPos = 82;
			gfComment = 0;

			executeTCode();
			gfTibMax = oldTibMax;	
			gfTibPos = oldTibPos;
			return gfErrors;							//=0 if everything OK
		}
        else  {											//VARCON
			gfSP++;
			*gfSP = gfUserHeader[gfParsed].toCompile;	//special;
			return 0;									//everything OK
		}
	}
	else {												// compiling
		compileTCode(gfParsed | 0x80); 
								//18Jan25check for A_VARCON in executeTCode()
		return gfErrors;								//=0 if everything OK
	}
	return gfErrors;						//this exit should be impossible;
}


uint32_t compileUserSymbol(uint32_t attribute)   // *************************
// **************************************************************************
{
uint32_t n;
uint8_t userString[8];
const uint8_t warnStr [] = "Warn: overrides Kernel OP w.same name ";

	if (gfNextNameCompi == MAXUSER)  {
		gfErrors |= ERR_NUMVAL;
		return 0;
	}
	gfTokenStart = gfTibPos;
	scanNext();
	gfTokenStart = gfTibPos;

   if (interpretNumber() != 0)  {
        gfErrors |= ERR_CONTEXT;
        return 0;
    }

	if (findKernelToken())  {
	    sendString(warnStr);
		sendTB(CRR);
        if (conFirm() == 0)  return 2;    //not compiled, no error flag set
	}

    for (n = 0; n < 8;n++) {
		userString[n] = gfTokenstr[n];
//transform into capital letters:
		if ((userString[n] >='a')&&(userString[n] <='z'))userString[n] -=0x20;
		if (userString[n] == 0) break;
		if (n == 7)  {
			gfErrors |= ERR_SYNTAX;	//name string too long
								//changed because bugfix26May22,see scanNext
			return 0;
		}
    }
    n = 0;
    while ((userString[n] != 0) && (n < 7))  {
      gfUserHeader[gfNextNameCompi].name[n] = userString[n];
      n++;
    }
    gfUserHeader[gfNextNameCompi].name[n+1] = 0;
	gfUserHeader[gfNextNameCompi].toCompile=(uint32_t)(&gfTCode[gfCodeIndex]);
	gfUserHeader[gfNextNameCompi].attributes = gfCodeIndex | attribute;
	return 1;
}


void compileTCode(uint32_t token)   // *************************************
// *************************************************************************
{
	gfTCode[gfCodeIndex] = (uint8_t)(token & 0x000000FF);
	gfCodeIndex ++;
//bugfix 20Dec23:
	if (gfCodeIndex >= CODETOP)  gfErrors |= ERR_USERCODE;
}

void executeTCode(void)   // ***********************************************
// *************************************************************************
{
uint32_t tibstate;
uint32_t index;

const uint8_t finiStr [] = " debug finished ";

    while (gfIP < gfTCode + USERTOP)  {
		if (gfBackProcess == 0)  {
			if (*gfIP < KERNELSIZE) gfKernelOps[*gfIP]();	//is Kernel Token
			else {								//then it is a User Token
				index = *gfIP & 0x0000007F;		//new 18Jan25:
				if (gfUserHeader[index].attributes & A_VARCON)  {	
					gfSP++;
					*gfSP = gfUserHeader[index].toCompile;
				}
				else  intoUserCode();
				}
		}
		else  {
			if (checkTerminal())  {
				tibstate = subQuery();
				if (tibstate == 1)  {			//Terminal CRR input
					backProcEval();
				}
				if (tibstate == 2)  {			//Terminal ESC input
					gfErrors |= ERR_CANCEL;
					return;
				}
//tibstate = 3(TAB=step) ignored in this case
			}
			if (*gfIP < KERNELSIZE)  {			//is Kernel Token
				if (gfBackProcess == 3)  execNext();
				else  gfKernelOps[*gfIP]();
			}
			else {								//then it is a User Token
				index = *gfIP & 0x0000007F;		//new 18Jan25:
				if (checkVarcon(index)) intoUserCode();
			}
		}

#ifndef USARTIO
//serve HW Background process
		if (gfStepSpeed)  {
			if ((gfUserHeader[4].toCompile > 0) ||
								(gfUserHeader[4].toCompile == 0xFFFFFFFE))  {
				if (gfSTEPCOUNT == 0) subMOT();
			}
		}
#endif
		if (gfErrors) return;
		if (gfIP)  {			//modified 29Jan25
			gfIP ++;
			continue;
		}
		else {					//execution finished by SEMIS or RETURN
			if (gfBackProcess == 3) sendString(finiStr);
			if (gfBackProcess)  gfBackProcess = 1;	//Bugfix 26Jan25
			gfExecUser = 0;
			gfBackNumValid = 0;
			gfNestLevel = 0;
			return;
		}
	}
}


void intoUserCode(void)   // ***********************************************
// *************************************************************************
{
uint32_t index;

	gfNestLevel++;
	gfRP ++;
	*gfRP = (uint32_t)gfIP;						//gets restored by "op_SEMIS"
	index = *gfIP & 0x0000007F;
	gfIP = (uint8_t*)gfUserHeader[index].toCompile;

	if (*gfIP < KERNELSIZE)  {					//is Kernel Token  
		if (gfBackProcess != 3)  gfKernelOps[*gfIP]();
		else  execNext();
	}
	else  {										//then it is a User Token
		index = *gfIP & 0x0000007F;				//new index in next UserToken!
		if (checkVarcon(index)) intoUserCode;		//new 18Jan25

	}
}

uint32_t checkVarcon(uint32_t index)   // ***********************************
//new 18Jan25
// *************************************************************************
{
uint32_t result = 0;

	if (gfUserHeader[index].attributes & A_VARCON)  {	
		gfSP++;
		*gfSP = gfUserHeader[index].toCompile;
		result =  0;
	}
	else  result = 1;
	if (gfBackProcess == 3)  {
		showPreStep(index+1);	//safety against index=0
		if (gfErrors) return result;
		showPostStep();
	}
	return result;
}


void execNext(void)   // ***************************************************
//abbreviation to save runtime if not in "step mode"
// *************************************************************************
{
    showPreStep(0);
    if (gfErrors) return;		//usually ESC = terminate User Thread
    gfKernelOps[*gfIP]();
    showPostStep();
}


void showPreStep(uint32_t index)  // ****************************************
// First the name of the next code token is sent to terminal.
// Execution of this token is triggered by TAB key on terminal software.
// Finally (by "showPostStep") the result is sent to terminal (output + stack)
// can be used as a simple Forth-Decompiler, too
// to speed display up, keep TAB key pressed
// **************************************************************************
{
uint32_t k, ipVal, indent, ipNext, data;
uint32_t *localIP;
//uint8_t *nameAddr;
const uint8_t IPStr [] = "(IP->";

    if (gfErrors)  return;

	gfIPold = gfIP;							//keep in mind for showPostStep()
    ipVal = *gfIP;
    indent = gfNestLevel * 3;

    sendTB(CRR);
    for (k=0; k < indent; k++) sendTB(SPACE);
    sendNX32((uint32_t)gfIPold & 0x0000FFFF);
    sendTB(SPACE);

	if (index)  {					//new 18Jan25
		index -= 1;				//was incremented to fit for userHeader[0]
		index2String(index);
		sendTB(0x28);	 //'('
		sendTB('=');
		if (gfTempNumBase==16)  send0X32(gfUserHeader[index].toCompile);
		else  sendNS32(gfUserHeader[index].toCompile);
		sendTB(0x29);	 //')'
	}
	else  {
		sendNX32 (ipVal & 0x0000FFFF);
		sendTB(SPACE);
		if (ipVal > KERNELSIZE)  {			//user token
			sendTB('-');
			sendTB('>');
			sendTB(SPACE);
		}
		token2String(ipVal);
	}

    if (((ipVal == IDX_DOIF) && (*gfSP == 0)) ||
			((ipVal == IDX_DOCASE) && (*gfSP != *(gfSP-1))) ||
			((ipVal == IDX_DOLTCAS) && (*(gfSP-1) >= *gfSP)) ||
			((ipVal == IDX_DOGTCAS) && (*(gfSP-1) <= *gfSP)) ||
			(ipVal == IDX_DOELSE) ||
			((ipVal == IDX_DOUNTIL) && (*gfSP == 0)) ||
			((ipVal == IDX_DOWHILE) && *gfSP) ||
			(ipVal == IDX_DOAGAIN) || 
			(ipVal == IDX_DOLOOP))  {
		sendString(IPStr);
		sendNX32(((*(gfIP+1)<< 8) | *(gfIP+2)) + 1);
                        //gfIP is incremented before "executeTCode()" is left
        sendTB(0x29);	                    //')'
    }

    if (ipVal==IDX_DOLIT8B) {
		ipNext = (uint32_t)(*(gfIP+1));
        sendTB(0x28);	                    //'('
        if (gfTempNumBase==16)   send0X32(ipNext);
        else  sendNS32(ipNext);
        sendTB(0x29);	                    //')'
    }

    if (ipVal==IDX_DOLIT) {
		ipNext = (uint32_t)(*(gfIP+1) << 24);
		ipNext |= (uint32_t)(*(gfIP+2) << 16);
        ipNext |= (uint32_t)(*(gfIP+3) << 8);
        ipNext |= (uint32_t)*(gfIP+4);
        sendTB(0x28);	                    //'('
        if (gfTempNumBase==16)   send0X32(ipNext);
        else  sendNS32(ipNext);
        sendTB(0x29);	                    //')'
    }
    ipNext = (uint32_t)(*(gfIP + 1) & 0x7F);
    if (ipVal==IDX_DOWVA) {
        sendTB(0x28);	                    //'('
		index2String(ipNext);
        sendTB('@');
		data = (uint32_t)&gfUserHeader[ipNext].toCompile;
        send0X32(data & 0x0000FFFF);
        sendTB(0x29);	 //')'
    }

    if (ipVal==IDX_SEMIS) {
        sendTB(SPACE);
        sendTB('<');
        sendTB('-');
    }

	k = 0;
	while (k != 3)  {			//while terminal input != TAB
		if (gfBackProcess < 3) break;
		servSystem();
		if (checkTerminal())  {
			k = subQuery();
			if (k == 1)   {		//Terminal CRR input
				backProcEval();
				continue;	//background operation does not release step
			}
			if (k == 2)  {		//Terminal ESC input
				gfErrors |= ERR_CANCEL;
				return;
			}
//new 31Dec23:
			if (k == 4)  {		//Terminal CTRL_R input
				if (gfNestLevel != 0)  {
					gfBackNestLevel = gfNestLevel;
					gfBackProcess = 2;
				}
				else  k = 3; //continue;
			}
		}
	}
	sendTB(SPACE);
	if ((ipVal == IDX_CONTI) || (ipVal == IDX_BREAK))  {
		sendString(IPStr);
	}
	else  {
		sendTB('{');
		sendTB(SPACE);
	}
	return;
}


void showPostStep(void)  // ************************************************
// remark see "showPreStep()"
// *************************************************************************
{
uint32_t n, ipVal;

	if (gfErrors)  return;

    ipVal = *gfIPold;

	if ((ipVal == IDX_CONTI) || (ipVal == IDX_BREAK))  {
		sendNX32(((uint32_t)(gfIP+1)) & 0x00000FFF);
                        //gfIP is incremented before "executeTCode()" is left
        sendTB(0x29);	
		return;
	}
	sendTB('}');
    sendTB(SPACE);
    op_PRINTSTACK();

    if ((ipVal >= IDX_DODO) && (ipVal <= IDX_RP4DEC))  {
        sendTB(SPACE);
        sendTB('R');
        sendTB(':');
    }
    if ((ipVal == IDX_DOFOR) || (ipVal == IDX_DOLOOP) ){
        if (((uint32_t)(gfRP - gfReturnStack) > 2)) {
            sendNX32((*(gfRP-3) & 0x0000FFFF));
            sendTB(SPACE);
            sendTB(SPACE);
            sendNX32((*(gfRP-2) & 0x0000FFFF));
            sendTB(SPACE);
            sendTB('i');
            sendNX32((*(gfRP-1) & 0x0000FFFF));
            sendTB(SPACE);
            sendTB('x');
            sendNX32((*gfRP & 0x0000FFFF) + 1);
			//the compiled jump address is 1 less than effectively executed
        }
    }
    else if ((ipVal >= IDX_DODO) && (ipVal <= IDX_DOAGAIN))
                                            sendNX32(*gfRP & 0x0000FFFF);

}


void emptyForth(void)// ****************************************************
//load and install an empty project with default settings
//Two CONSTants P0, P1 and two VARs V2, V3 are installed as user token
// *************************************************************************
{
uint32_t k;
//uint32_t *pointer;

	gProject = 0;
    gfBackProcess = 0;					//default mode changed 26Jan25
    gfAuto = 0;	
	RCC->CFGR &= ~(RCC_CFGR_HPRE);		//clear all HPRE options
	RCC->CFGR |= RCC_CFGR_HPRE_DIV1;	//and replace

//default SPI setup:
	gfSpiMode = 0;
	gfSpiBits = 8;
	gfSpiClock = 50;	//default:about 50kHz clock

//default PWM setup:
	gfFreq = 0;			//default = max = :24000 Hz)
    gfPWM1 = -1;		//-1 is OFF, PWM max 2000)
    gfPWM2 = -1;		//-1 is OFF

//compile 4 VARCONs for general use
	for (k = 0; k < 4;k++) {
		gfUserHeader[k].name[0] = 'V';
		gfUserHeader[k].name[1] = k + 0x31;
		gfUserHeader[k].name[2] = 0;
		gfUserHeader[k].attributes = A_VARCON;
		gfUserHeader[k].toCompile = 0;
	}	
#ifndef USARTIO
	gfStepSpeed = 0;
//compile VARCON NSTP(number of steps) for op_MOT background control
	gfUserHeader[4].name[0] = 'N';
	gfUserHeader[4].name[1] = 'S';			//bugfix 01Jan24
	gfUserHeader[4].name[2] = 'T';			//confused with op_MS
	gfUserHeader[4].name[3] = 'P';
	gfUserHeader[4].name[4] = 0;
	gfUserHeader[4].attributes = A_VARCON;
	gfUserHeader[4].toCompile = -1;		//start with DC motor
	gfNextNameCompi = 5;
	gfUserHeader[5].name[0] = 0;
#else
	gfNextNameCompi = 4;
	gfUserHeader[4].name[0] = 0;
#endif
	gfCodeIndex = 0;
	op_ABORT();
}


// arithmetic ops  #########################################################

void op_ERR(void)  // *****************************************************
// ( -- )
// does nothing, except consume a little bit of runtime
// *************************************************************************
{
	gfErrors |= ERR_USERCODE;
}

void op_DROP(void)  // *****************************************************
// ( x1 x2 -- x1)
// forget the TOS and pop all lower stack entries one up
// *************************************************************************
{
	gfSP--;
	if (gfSP < gfDataStack + 3) gfErrors |= ERR_STACK;
}


void op_DUP(void)  // ******************************************************
// ( x -- x x)
// push all stack entries one down and copy the new NOS as TOS
// *************************************************************************
{
	gfSP++;
	if (gfSP >= gfDataStack+MAXDATASTACK - 4)  {
        gfErrors |= ERR_STACK;
        return;
	}
	*gfSP = *(gfSP-1);
}


void op_SWAP(void)  // *****************************************************
// (x1 x2 -- x2 x1)
// exchange TOS and NOS
// *************************************************************************
{
uint32_t TEMP;

	TEMP = *(gfSP-1);
	*(gfSP-1) = *gfSP;
	*gfSP = TEMP;
}


void op_OVER(void)  // *****************************************************
// ( x1 x2 -- x1 x2 x1)
// push all stack entries one down and copy the new 3rd of stack to TOS
// *************************************************************************
{
	gfSP++;
	*gfSP = *(gfSP-2);
}


void op_ROT(void)  // ******************************************************
// ( x1 x2 x3 -- x2 x3 x1 )
// rotate 3rd stack entry up
// *************************************************************************
{
uint32_t TEMP;

	TEMP = *(gfSP-2);
	*(gfSP-2) = *(gfSP-1);
	*(gfSP-1) = *gfSP;
	*gfSP = TEMP;
}


void op_PICK(void)  // *****************************************************
// (  xn x(n-1) x2 x1 n  ---  xn x(n-1) ... x2 x1 x0 xn  )
// keep  n(=TOS) in mind and copy the (n+1)th element of stack on TOS
// 0 PICK performs DROP
// 1 PICK effectively performs DUP
// 2 PICK effectively performs OVER
// *************************************************************************
{
int32_t temp;

//first check available depth of stack
    temp = (int32_t)(gfSP - gfDataStack - 4) - (int32_t)*gfSP;
    if  (temp < 0) {
        gfErrors |= ERR_STACK;
        return;
    }
	temp = *gfSP;
	if (temp == 0) {
		gfSP--;
		return;
	}
	*gfSP = *(gfSP - temp);
}


void op_ROLL(void)  // *****************************************************
// (  xn x(n-1)...x2 x1 n --- x(n-1)  x2 x1 xn  )
// drops TOS, but keeps n in mind. Rotates the (new) n-th stack element 
// up to TOS. 
// 0 ROLL and 1 ROLL do nothing except dropping TOS
// 2 ROLL is equivalent to SWAP
// 3 ROLL is equivalent to ROT
// bigger values of "n" provide manipulation of deep stack items
// e.g. the sequence  "4 ROLL 4 ROLL" effectively performs 2SWAP
// *************************************************************************
{
uint32_t k, temp;

//first check available depth of stack
    k = (int32_t)(gfSP - gfDataStack - 4) - (int32_t)*gfSP;
    if  (k >= 0x80000000) {
        gfErrors |= ERR_STACK;
        return;
    }

	k = *gfSP;
	gfSP--;
	if (k < 2) return;
	temp = *(gfSP - k + 1); 
	while(k)  {
		*(gfSP - k + 1 ) = *(gfSP - k + 2);
		k -- ;
	}
	*gfSP = temp;
}


void op_INJECT(void)  // ****************************************************
// (  x(n+1) xn... x2 x1 n  ---  x1 x(n+1) xn... x2  )
// drops TOS, keeps n in mind. 
// then rotates (new) TOS(= x1) down below the (actual) n-th stack element.
// 0 INJECT and 1 INJECT simply drop TOS
// 2 INJECT effectively performs SWAP
// 3 INJECT effectively performs -ROT
// bigger values of "n" provide manipulation of deep stack items
// e.g.sequence "n PICK n INJECT" effectively doubles the deep n-th stack item
// **************************************************************************
{
uint32_t k, n, temp;

//first check available depth of stack
    k = (int32_t)(gfSP - gfDataStack - 4) - (int32_t)*gfSP;
    if  (k >= 0x80000000) {
        gfErrors |= ERR_STACK;
        return;
    }

	temp = *(gfSP - 1);
	n = *gfSP;
	gfSP --;
	if (n < 2) return;
	for (k=0;k < (n-1);k++)  {
		*(gfSP - k) = *(gfSP - k - 1);
	}
	*(gfSP - n + 1) = temp;
}


void op_PLUS(void)  // *****************************************************
// ( x1 x2  -- x1+x2)
// *************************************************************************
{
	*(gfSP-1) = (int32_t)*(gfSP-1) + (int32_t)*gfSP;
	gfSP--;
}


void op_INCT(void)  // *****************************************************
// ( x -- x+1)
// *************************************************************************
{
	*gfSP = *gfSP + 1;
}


void op_MINUS(void)  // ****************************************************
// ( x1 x2  -- x1-x2)
// *************************************************************************
{
	*(gfSP - 1) = (int32_t)*(gfSP - 1) - (int32_t)*gfSP;
	gfSP--;
}


void op_DECT(void)  // *****************************************************
// ( x -- x-1)
// *************************************************************************
{
	*gfSP = *gfSP - 1;
}


void op_ABS(void)  // ******************************************************
// ( x -- abs(x))
// *************************************************************************
{
int32_t abs;

	abs = (int32_t)*gfSP;
	if (abs < 0) *gfSP = -abs;
}


void op_SHIFT(void)  // ***************************************************
// if (X2>=0) ( x1 x2 -- x1<<x2 )
// if (X2<0) ( x1 x2 -- x1>>x2 )
// abs(x2) <= 31
// *************************************************************************
{
int32_t k,abs;

	k = (int32_t)*gfSP;
	abs = k;
	if (abs < 0) abs = -abs;
		if (abs > 31)  {
			gfErrors |= ERR_NUMVAL;
			return;
		}
	gfSP--;
	if (k >= 0)   {
		*(gfSP) = *(gfSP)<<k;
	}
	else  {
		*(gfSP) = *(gfSP)>>abs;
	}

}


void op_AND(void)  // ******************************************************
// ( x1 x2  -- x1&x2)
// *************************************************************************
{
	*(gfSP-1) &= *gfSP;
	gfSP--;
}

void op_OR(void)  // ******************************************************
// ( x1 x2  -- x1|x2)
// *************************************************************************
{
	*(gfSP-1) |= *gfSP;
	gfSP--;
}


void op_XOR(void)  // ******************************************************
// ( x1 x2  -- x1 XOR x2)
// *************************************************************************
{
	*(gfSP-1) = *gfSP ^ *(gfSP-1);
	gfSP--;
}


void op_NOT(void)  // *****************************************************
// ( x --- <one's complement of x>)
// logical inversion bit by bit (one's complement)
// in contrast to other Forth,this is an arithmetic operator differing from "0="
// *************************************************************************
{
	*gfSP = ~ (*gfSP);
}


void op_NEG(void)  // *****************************************************
// ( x --- <two's complement of x>)
// mathematical negation (two's complement)
// *************************************************************************
{
	*gfSP = -(int32_t)(*gfSP);
}


void op_MULT(void)  // *****************************************************
// ( factor factor  -- TOS=factor*factor)
// multiplication is made 32bit signed-i.e. may overflow with factors>0xB000
// if this case: no error msg, no warning. User is responsible
// *************************************************************************
{
	*(gfSP-1) = (int32_t)*gfSP * (int32_t)(*(gfSP-1));
	gfSP--;
}


void op_DIV(void)  // ******************************************************
// ( dividend divisor  -- x1/x2)
// 32 bit signed division
// *************************************************************************
{
    if (*gfSP == 0)  {
        gfErrors |= ERR_DIVZERO;
        return;
    }
	else  {
		*(gfSP-1) = (int32_t)*(gfSP-1) / (int32_t)*gfSP;
        gfSP--;
	}
}


void op_MULTDIV(void)  // **************************************************
// ( factor factor divisor-- TOS=(factor*factor)/divisor)
// multiplication is made 32bit int: i.e. may overflow with factors>0xB000
// if this case: no error msg, no warning. User is responsible
// *************************************************************************
{
int32_t factor, dividend, divisor;

    if (*gfSP == 0)  {
        gfErrors |= ERR_DIVZERO;
        return;
    }
	else  {
        factor = (int32_t)*(gfSP - 2);
        dividend = (int32_t)*(gfSP-1);
        divisor =  (int32_t)*gfSP;
        dividend = factor * dividend;
		gfSP -= 2;
        * gfSP = dividend / divisor;
	}
}


void op_MOD(void)  // ******************************************************
// return the remainder of a 32 bit signed integer division
// ( dividend divisor  -- remainder )
// the remainder has always the same sign as the dividend
// *************************************************************************
{
    if (*gfSP == 0)  {
        gfErrors |= ERR_DIVZERO;
        return;
    }
	else  {
		*(gfSP-1) = (int32_t)*(gfSP-1) % (int32_t)(*gfSP);
		gfSP--;
	}
}


void op_DIVMOD(void)  // ******************************************************
// return the remainder(NOS) and quotient(TOS) of a 32 bit signed int. division
// ( dividend divisor  -- remainder quotient )
// the remainder has always the same sign as the dividend
// *************************************************************************
{
int32_t dividend, divisor;

    if (*gfSP == 0)  {
        gfErrors |= ERR_DIVZERO;
        return;
    }
	else  {
        dividend = (int32_t)*(gfSP-1);
        divisor =  (int32_t)*gfSP;
		*(gfSP-1) = dividend % divisor;
		*gfSP = dividend / divisor;
	}
}

#ifndef USARTIO
void op_BSET(void)  // ****************************************************
// (oldTOS bit# -- newTOS) read modify write
// Top of Stack (TOS) manipulation
// bit# = bit number to be set (0 to 31)
// *************************************************************************
{
	if (*gfSP > 31) {
		gfErrors |= ERR_NUMVAL;
        return;
	}
    *(gfSP-1) |= 1 << *gfSP;        //bitNo-->bitMask and set NOS
	gfSP--;                         //NOS-->TOS
}


void op_BCLR(void)  // *****************************************************
// (oldTOS bit# -- newTOS) read modify write
// Top of Stack (TOS) manipulation
// bit# = bit number to be set (0 to 31)
// *************************************************************************
{
    if (*gfSP > 31) {
		gfErrors |= ERR_NUMVAL;
		return;
	}
    *(gfSP-1) &= ~(1 << *gfSP);   //bitNo-->bitMask and invert NOS bitwise
	gfSP --;                        //NOS-->TOS
}


void op_BTST(void)  // *****************************************************
// (oldTOS bit# -- oldTOS bitvalue)
// Top of Stack (TOS) manipulation
// bit# = bit number to be checked (0 to 31)
// new TOS = (bitvalue = boolean value of this bit (0 or 1)
// *************************************************************************
{
uint32_t temp;

	if (*gfSP > 31) {
		gfErrors |= ERR_NUMVAL;
		return;
	}

	temp = 1 << *gfSP;      //bitNo-->bitMask
    temp = *(gfSP-1) & temp;	
	if (temp) *gfSP = 1;
	else *gfSP = 0;
}
#endif

void op_RANDOM(void)  // ****************************************************
//  ( x1 x2  -- TRUE|FALSE)
//	TRUE if x1==X2
// *************************************************************************
{
	gfSP ++;
	*gfSP = gfRandom;
}


//comparison ops ###########################################################

void op_EQUAL(void)  // ****************************************************
//  ( x1 x2  -- TRUE|FALSE)
//	TRUE if x1==X2
// *************************************************************************
{
	if (*gfSP == *(gfSP-1)) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_UNEQU(void)  // ****************************************************
//  ( x1 x2  -- TRUE|FALSE)
//	TRUE if x1!=X2
// *************************************************************************
{
    if (*gfSP != *(gfSP-1)) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_ZEQ(void)  // ******************************************************
//  ( x   -- TRUE|FALSE)
//	TRUE if x==0, FALSE if x!=0
// *************************************************************************
{
	if (*gfSP == 0) *gfSP = 1;
	else *gfSP = 0;
}


void op_GRTR(void)  // *****************************************************
//  ( x1 x2  -- TRUE|FALSE)
//  signed compare
//	TRUE if x1>X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) > (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_GRTREQ(void)  // ***************************************************
//  ( x1 x2  -- TRUE|FALSE)
//  signed compare
//	TRUE if x1>=X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) >= (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_LESS(void)  // *****************************************************
//  ( x1 x2  -- TRUE|FALSE)
//  signed compare
//	TRUE if x1<X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) < (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_ULESS(void)  // *****************************************************
//  ( x1 x2  -- TRUE|FALSE)
//  unsigned compare
//	TRUE if x1<X2
// *************************************************************************
{
	if ((uint32_t)*(gfSP-1) < (uint32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


void op_LESSEQ(void)  // ***************************************************
//  ( x1 x2  -- TRUE|FALSE)
//  signed compare
//	TRUE if x1<=X2
// *************************************************************************
{
	if ((int32_t)*(gfSP-1) <= (int32_t)*gfSP) *(gfSP-1) = 1;
	else *(gfSP-1) = 0;
	gfSP-- ;
}


// memory ops  #############################################################

void op_VARCON(void)  // ****************************************************
// ( value -- )
// syntax example:
// 23 VARCON CONNY
// defines the VariableConstant CONNY and initializes it with value 23
// at runtime, a VARCON puts its value on TOS
// *************************************************************************
{
	if (compileUserSymbol(A_VARCON) == 0)  {
    	gfErrors |= ERR_CONTEXT;
    	return;
	}
	gfUserHeader[gfNextNameCompi].toCompile = *gfSP;
	gfSP--;
	gfNextNameCompi++;
	return;
}


void op_WVA(void)  // **************************************************
// set a new value to a VARCON
// special non-Forth typical syntax,
// example:
// 23 VARCON TEST
// 0xABCDE W TEST		//""W" is user level operator which triggers op_WVA
// (TEST must be scanned by W to get its .toCompile address)
// TEST .S   =returns 0xABCDE on stack
// the new value is inserted at any place where VARCON is threaded
// and will be used in future until a new value is inserted
// *************************************************************************
{
uint32_t addr;

	gfTokenStart = gfTibPos;
	scanNext();
	if (findUserToken() == 0) {		//updates "gfParsed"=index of gfUserHeader
		gfErrors |= ERR_USERCONTEXT;
		return;
	}
	gfTokenStart = gfTibPos;

    if ((gfUserHeader[gfParsed].attributes & A_VARCON) == 0)  {
 		gfErrors |= ERR_CONTEXT;
		return;
    }

	if (gfExecUser) checkBackNum();

	if (gfCompiState == 0)  {	    //interpreting
		gfUserHeader[gfParsed].toCompile = *gfSP; //re-init with stack value
		gfSP--;
	}
	else  {	    //compiles runtime "W" behaviour
	    compileTCode(IDX_DOWVA);
		compileTCode(gfParsed); 
	}
}


void op_ARRAY(void)	// *****************************************************
// 32bit signed integer array, 200 entries. index = 0 cannot be used
// negative TOS=index: write NOS into array[-index]
// positive TOS=index: read value from array[-index] to TOS
// write example  234 -100 array 
// read example  100 array p  returns 234
// *************************************************************************
{
int32_t index;

#ifdef USARTIO
static int32_t array[40];

	index = (int32_t)*gfSP;				//copy TOS to index
	if ((index == 0) || (index > 40) || (index < -40))  {
		gfErrors |= ERR_NUMVAL;
		return;
	}

#else

static int32_t array[200];

	index = (int32_t)*gfSP;				//copy TOS to index
	if ((index == 0) || (index > 200) || (index < -200))  {
		gfErrors |= ERR_NUMVAL;
		return;
	}
#endif
	if (index > 0)  {					//read array[-index] (-index --- value)
		*gfSP = array[index-1];			//adjust user index to array index
										//no stack pointer adjustment needed
	}
	else  {								//write NOS into array[-index]
										//TOS=index, NOS=value to be written
										// (value -index --- )
		index = - index;				//convert to effective index
		array[index-1] = *(gfSP-1);		//adjust user index to array index
										//(gfSP-1) is pointer to NOS
										//*(gfSP-1) is NOS value
		gfSP -= 2;						//adjust stack pointer:2 items removed
	}
}

// terminal ops  ###########################################################

void op_KEYQ(void)  // ****************************************************
//check if a character is received via serial terminal port (USART)
//if NOT return 0 on TOS
//if YES return(number of bytes in RX buffer) on TOS
//NO data byte is extracted from terminal input buffer
// *************************************************************************
{
    gfSP++;
    if (gfBackProcess && gfExecUser)  *gfSP = gfTermCheck;		
					//exclusively set '1' in subQuery(), cleared in op_KEY()
	else *gfSP = checkTerminal();
}


uint32_t checkTerminal(void)  // *******************************************
//check if a character is received via serial terminal port (USART)
//if NOT return 0
//if YES return number of bytes in RX buffer
// *************************************************************************
{
	if (gSTLINK_RXTAIL==gSTLINK_RXHEAD) return 0;
	else  {
		if (gSTLINK_RXTAIL < gSTLINK_RXHEAD)  {
			return ((uint32_t)(gSTLINK_RXHEAD - gSTLINK_RXTAIL));
		}
		else return (RXBUFLEN+(uint32_t)(gSTLINK_RXTAIL-gSTLINK_RXHEAD));
	}
}


void op_KEY(void)  // ******************************************************
//wait until a character is received via serial terminal port (USART)
//and return it on the stack
// *************************************************************************
{
uint32_t recvword;

    if (gfBackProcess && gfExecUser)  {
		if (gfTermCheck)  {
			gfSP++;
			*gfSP = gfTermChar;		//set in subQuery()
			gfTermCheck = 0;
		}
	}
	else  {
		while ((recvword = pullTermRxBuf())==0); 
									//watchdog is fed in the background
		gfSP++;
		*gfSP = recvword & 0x000000FF;
	}
}


void op_EMIT(void)  // *****************************************************
//send the lower 8 bits of TOS as BYTE!! via serial terminal port
// *************************************************************************
{
    sendTB((uint8_t)(*gfSP & 0x000000FF));
	gfSP--;
}


void op_PRINT(void)  // ****************************************************
//send the TOS number as ASCII text via serial terminal port ( USART)
//use system number base
//numbers in the range 0x20000000 to 0x50060C00 always are printed hex formatted
// *************************************************************************
{
    if ((gfTempNumBase == 16)||((*gfSP >= 0x20000000)&&(*gfSP < 0x50060C00))) {
         send0X32(*gfSP);
    }
    else sendNS32(*gfSP);
    sendTB(SPACE);
	gfSP--;
}


void op_PRINTHEX(void)  // *************************************************
//print the TOS number as hex style ASCII text via via serial terminal port
// *************************************************************************
{
    send0X32(*gfSP);
    sendTB(SPACE);
	gfSP--;
}


void op_PRINTDEC(void)  // *************************************************
//print the TOS number as hex style ASCII text via via serial terminal port
// *************************************************************************
{
	sendS32(*gfSP);;
	gfSP--;
}


void op_PRINTSTR(void)   // ************************************************
//compile the subsequent string to be sent at runtime via via serial terminal
//Note: between P" and the first character of the string MUST be a SPACE
//Non-printable characters (plus " and \) are entered as \+HexCode (w.o. 0x)
//essential examples \5C sends a Backslash, \20 sends a SPACE, \22 sends '"'
// *************************************************************************
{
    subPrintString(1);
}


void subPrintString(uint32_t out)   // *************************************
//out=1: send to Terminal, out=0: send to IO
//else remark see op_PRINTSTRING()
// *************************************************************************
{
uint32_t k, n, hibyte, lobyte, word;

	gfTokenStart = gfTibPos + 1;
	scanString();

	k= 0;
	n = 0;
	while (1)  {
		if (gfTokenstr[k] == 0x5C) n++; 	//Backslash '\'
						//each Backslash "transports" 3 bytes in Tokenstr[]
		if (gfTokenstr[k] == '"')  {
			gfTokenstr[k] = 0;
			break;
		}
		k++;
		if (k >= 80) {
			gfErrors |= ERR_SYNTAX;
			return;
		}
	}
	if (gfCompiState) {							//compiling
#ifdef USARTIO
		if (out == 1)  compileTCode(IDX_DOSTR);
		else  compileTCode(IDX_DOIOSTR);
#else
		compileTCode(IDX_DOSTR);
#endif
		gfTCode[gfCodeIndex]= k - (n * 2);	//string length count
		gfCodeIndex ++;
		k = 0;
		do {
			if (gfTokenstr[k] != 0x5C)  {	//not Backslash '\'
				compileTCode(gfTokenstr[k]);
				k++;
			}
			else  {
				lobyte = packBackslash(k);
				if (lobyte == 0xFFFFFFFF)  {
					gfErrors |= ERR_SYNTAX;
					return;
				}
				k += 3;
				compileTCode(lobyte);
			}
		} while (gfTokenstr[k] != 0);
	}
	else  {									//interpreting
		k = 0;
		do {
			if (gfTokenstr[k] != 0x5C)  {      //'\'
#ifdef USARTIO
				if (out == 1) sendTB(gfTokenstr[k]);
				else if (gSplit) sendIOB(gfTokenstr[k]);
						//else ignore silently
#else
				sendTB(gfTokenstr[k]);
#endif
				k++;
			}
			else  {
				lobyte = packBackslash(k);
				if (lobyte == 0xFFFFFFFF)  {
					gfErrors |= ERR_SYNTAX;
					return;
				}
				k += 3;
#ifdef USARTIO
				if (out == 1) sendTB(lobyte);
				else if (gSplit) sendIOB(lobyte);
						//else ignore silently
#else
				sendTB(lobyte);
#endif
			}
		} while (gfTokenstr[k] != 0);
	}
}


void scanString(void)   // ********************************************
// ************************************************************************
{
uint32_t k;

    k=0;
	while (gfTib[gfTokenStart+k] != 0x22)  {
                                    //search trailing " after scanned token
		gfTokenstr[k] = gfTib[gfTokenStart+k];
		k++;
		if (k >= 78) break;  //limit length of string
	}
	gfTokenstr[k] = 0x22;	//Apostroph
	k++;
	gfTokenstr[k] = 0;
	gfTibPos = gfTokenStart + k;
	return;
}


uint32_t packBackslash(uint32_t k)   // ***********************************
// ************************************************************************
{
uint32_t hibyte, lobyte;
	k++;
	hibyte = gfTokenstr[k];
	if ((hibyte >= 'a') && (hibyte <= 'z')) hibyte -= 0x20;
	if (hibyte < 0x30)  return 0xFFFFFFFF;
	hibyte -= 0x30;
	if (hibyte > 9) hibyte -= 7;
	if (hibyte > 0x0F)  return 0xFFFFFFFF;
	k++;
	lobyte = gfTokenstr[k];
//	k++;
	if ((lobyte >= 'a') && (lobyte <= 'z')) lobyte -= 0x20;
	if (lobyte < 0x30)  return 0xFFFFFFFF;
	lobyte -= 0x30;
	if (lobyte > 9) lobyte -= 7;
	if (lobyte > 0x0F)  return 0xFFFFFFFF;
	lobyte += hibyte * 16;
	return lobyte;
}


void op_DECIM(void)  // ****************************************************
//set the system number base to DECIMAL
//Forth internal operations use gTempNumBase, which is usually=gNumBase
//if op_Decim is called during compilation, only gTempNumBase is changed
//op_Semis() or the error handler restores gTempNumBase to gNumBase 
//during compilation number base may temporarily change without global change
// *************************************************************************
{
	gfTempNumBase = 10;
	if (gfCompiState== 0) gfNumBase = 10;
}


void op_HEX(void)  // ***********************************************
//set the system number base to HEX
//comment see op_DECIM
// *************************************************************************
{
	gfTempNumBase = 16;
	if (gfCompiState== 0) gfNumBase = 16;
}

#ifdef USARTIO
void op_RXQ(void)  // ******************************************************
//check if a character is received via serial I/O (contained in the RX buffer)
//if NOT return 0 on TOS
//if YES return(number of bytes in RX buffer) on TOS
//for MIDI I/O: number of MIDI Messages (except SysEX) is returned
//SPLIT assignment:
//'0'=USB and USART both terminal,
//'1'=USB is terminal, USART is COM I/O port
//'3'=USB is terminal, USART is MIDI interface
// *************************************************************************
{
   if (gSplit == 1)  {
        gfSP++;
        if (gUSART_RXTAIL==gUSART_RXHEAD) *gfSP=0;
        else  {
            if (gUSART_RXTAIL < gUSART_RXHEAD)
                *gfSP = (uint32_t)(gUSART_RXHEAD - gUSART_RXTAIL);
            else *gfSP=RXBUFLEN+(uint32_t)(gUSART_RXTAIL-gUSART_RXHEAD);
        }
    }
    else if (gSplit == 3)  {
		Rx2Midi(); //check if something in the low-level queue and pack->Midi
        Rx2Midi(); //try to pack a complete message
		Rx2Midi();
        gfSP++;
        if (gMIDI_TAIL==gMIDI_HEAD)  *gfSP=0;
        else  {
            if (gMIDI_HEAD > gMIDI_TAIL)  *gfSP = gMIDI_HEAD - gMIDI_TAIL;
            else  *gfSP = MIDI_BUFLEN - (gMIDI_TAIL - gMIDI_HEAD);
        }
    }
    else  gfErrors |= ERR_COMMACC;
}


void op_RX(void)  // ******************************************************
//wait until a character is received via serial I/O and return it on the stack
//for MIDI I/O: MIDI Messages (except SysEX) are returned as special words
//SPLIT assignment:
//'0'=USB and USART both terminal,
//'1'=USB is terminal, USART is COM I/O port
//'3'=USB is terminal, USART is MIDI interface
// *************************************************************************
{
uint32_t recvword;

	if (gSplit == 1)  {
		while ((recvword = pullUsartRxBuf()) == 0)  {
									//watchdog is fed in the background
			if (checkBreak()) return;
		}
		gfSP++;
		*gfSP = recvword & 0x000000FF;
	}
	else if (gSplit == 3)  {
		while ((recvword = pullMidiBuf()) == 0)  {
		//pullMidiBuf responds with != 0 when a complete MIDI msg is received
		//servSystem() is processed in the background
			Rx2Midi(); //check if something in the lowlevel queue and packMidi
			if (checkBreak()) return;
		}
		gfSP++;
		*gfSP = recvword & 0x00FFFFFF;
	}
	else  gfErrors |= ERR_COMMACC;
}


void Rx2Midi(void)  // **************************************************
//transfer bytes from serial RX buffer to MidiScanner()
//is performed implicitly in RX?, RX, query()
//explicit call by user only necessary in special situations
// *************************************************************************
{
uint32_t recvword;

	if (gSplit == 3)  {
		recvword = pullUsartRxBuf();  //USART is MIDI IN
		if (recvword)  MidiScanner(recvword & 0x000000FF);
	}
}


void op_TX(void)  // *******************************************************
//send the lower 8 bits of TOS as BYTE!! via serial I/O
// *************************************************************************
{
    sendIOB((uint8_t)(*gfSP & 0x000000FF));
	gfSP--;
}


void op_NTX(void)  // ************************************************
//send TOS times the lower 8 bits of next stack entries as BYTE!! via serial I/O
//example: 0x90 0x33 0x40 3 NTX sends a MIDI note
// *************************************************************************
{
uint32_t k,n;

    k = *gfSP;
    n = k;
    gfSP--;
    while (n)  {
        sendIOB((uint8_t)(*(gfSP - n + 1) & 0x000000FF));
        n--;
    }
    gfSP -= k;
}


void op_TXSTR(void)   // ***************************************************
//compile the subsequent string to be sent at runtime via serial I/O
//else see remark at op_PRINTSTRING()
// *************************************************************************
{
    subPrintString(0);
}
#endif


//time ops  ################################################################

void op_XNOP(void)  // *****************************************************
// ( delay -- )
// start a very short blocking delay
// duration is modified by stack parameter
// example for a minimal loop
// : TEST 0xA0 DO DUP OH 0 XNOP DUP OL 100 XNOP AGAIN;
// measured under no-debug condition:
// the HIGH period takes about 500us,the LOW period about 1400us(921kHz clock)
// the HIGH period takes about 90us,the LOW period about 300us(3,68M clock)
// the HIGH period takes about 20us,the LOW period about 75us(14M clock)
// with backgound process active:
// the HIGH period takes about 800us,the LOW period about 1700us(921kHz clock)
// the HIGH period takes about 170us,the LOW period about 400us(3,68M clock)
// the HIGH period takes about 40us,the LOW period about 100us(14M clock)
// *************************************************************************
{
uint32_t loops;

    loops = *gfSP;		//bugfix 17June22, stack pointer syntax error
	gfSP--;
	while (loops) {
        loops--;
	}
}


void op_MS(void)  // *******************************************************
// ( milliSeconds -- )
//blocks CForth execution for *gfSP milliSeconds
// example
// 1000 MS   blocks for 1 second
// *************************************************************************
{
uint32_t tibstate;

	gfMSCOUNT = *gfSP;		//milliseconds
	gfSP--;
	while (gfMSCOUNT)  {
		servSystem();
#ifndef USARTIO
//serve Background process
		if (gfStepSpeed)  {
			if ((gfUserHeader[4].toCompile > 0) ||
								(gfUserHeader[4].toCompile == 0xFFFFFFFE))  {
				if (gfSTEPCOUNT == 0) subMOT();
			}
		}
#endif
		if (checkBreak()) return;
	}
}


int32_t checkBreak(void)  // **********************************************
// *************************************************************************
{
uint32_t tibstate;

	if (gfTermInput == CTRL_C) {		//CONTINUE prog flow, ignore countdown
		gfTermInput = 0;
		return 1;
	}
	if (gfBackProcess && gfExecUser)  {
		if (checkTerminal())  {
			tibstate = subQuery();
			if (tibstate == 1)  backProcEval();	//terminal CRR input
			if (tibstate == 2)  gfErrors |= ERR_CANCEL;
			if (gfErrors)  return 1;
		}
	}
	return 0;
}


void op_DECIS(void)  // ****************************************************
// ( deciSeconds -- )
//blocks CForth execution for *gfSP deciSeconds
// example
// 10 DCS   blocks for 1 second
// *************************************************************************
{
	*gfSP *= 100;		//deciSeconds = milliSeconds * 100;
	op_MS();
}


void op_TIX(void)  // ******************************************************
// ( time -- )
// start a countdown measured in tenths of seconds.
// countdown is controlled by WAIT (no-blocking) or BLOCK (blocking)
// *************************************************************************
{
    gfTIXCOUNT = *gfSP *100;       // internally scaled in milliseconds
	gfSP--;
}


void op_TIME(void)  // *****************************************************
// ( -- time)
//returns remaing time in tenths of seconds while "MS or "TIX" is not finished
//else returns 0 after "MS od "TIX" has finished
// *************************************************************************
{
	gfSP++;
    *gfSP = gfTIXCOUNT/100;
}


void op_FREQ(void)  //  ****************************************************
// ( freq -- )
//start peripheral TIMER3 to work on PA6 and/or PA7 as output
// *************************************************************************
{
	if (gfExecUser) checkBackNum();

	gfFreq = *gfSP;
    gfSP --;
	subFREQ();;
    subPWM1();
//#ifndef USARTIO
    subPWM2();
//#endif
}


void subFREQ(void)  // ****************************************************
// ( freq -- )
//start peripheral TIMER3 to work on PA6 and/or PA7 as output
// *************************************************************************
{
	if (gfFreq > 0x0000FFFF)  gfFreq = 0x0000FFFF;

	if (gfFreq)  {										//start timer
		if ((TIM3->CR1 & TIM_CR1_CEN) == 0)  {	//not yet initialized
			TIM3->SR = 0;
			TIM3->CR1 = 0;
			RCC->APB1ENR |= RCC_APB1ENR_TIM3EN;
			TIM3->CCMR1 = (6<<12) | (6<<4);
								//Channel1 and 2,each PWM mode1,compare output
			TIM3->ARR = 2000;					//about 11 bit resolution
												//but easier calculation
												//freq=24 gives 1kHz pulse
			TIM3->CR1 |= TIM_CR1_CEN;
		}
//initial or restart of timer:
		TIM3->PSC = gfFreq - 1;
		//PSC is 16 bit register, Timer3 freq about 921.6kHz down to ca 0.1Hz
		//1 is added in TIM3_PSC prescaler divisor
		if (gfPWM1 < 0) TIM3->CCER &= ~TIM_CCER_CC1E;//disable output handling
		else TIM3->CCER |= TIM_CCER_CC1E;			//enable output handling
#ifndef MOTOR
		if (gfPWM2 < 0) TIM3->CCER &= ~TIM_CCER_CC2E;//disable output handling
		else TIM3->CCER |= TIM_CCER_CC2E;			//enable output	handling
#endif
	}
	else  {										//gfFREQ = 0 --> Timer OFF
		TIM3->CR1 &= ~TIM_CR1_CEN;				//disable timer
		gfPWM1 = -1;
		GPIOA->MODER &= ~(GPIO_MODER_MODER6);	//Set PA6 as input(default)
		GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR6;
		GPIOA->PUPDR |= GPIO_PUPDR_PUPDR6_1;	//pull-down
		TIM3->CCER &= ~TIM_CCER_CC1E;			//disable output handling
#ifndef MOTOR  //PWM2 is OFF
		gfPWM2 = -1;
		GPIOA->MODER &= ~(GPIO_MODER_MODER7);	//Set PA7 as input(default)
		GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR7;
		GPIOA->PUPDR |= GPIO_PUPDR_PUPDR7_1;	//pull-down
		TIM3->CCER &= ~TIM_CCER_CC2E;			//disable output handling
#endif
	}
}


void op_PWM1(void)  // ******************************************************
// ( pwm -- )
//start peripheral TIMER3, channel 1 to work on PA6 as output
// *************************************************************************
{
	if (gfExecUser) checkBackNum();

	gfPWM1 = (int32_t)*gfSP;
	gfSP --;

	if ((TIM3->CR1 & TIM_CR1_CEN) == 0)  {		//timer emergency activation
		gfFreq = 24;
        subFREQ();
	}
	subPWM1();
}


void subPWM1(void)  // *****************************************************
// ( pwm -- )
//start peripheral TIMER3, channel 2 to work on PA6 as output
// *************************************************************************
{
	GPIOA->MODER &= ~(GPIO_MODER_MODER6);	//temporarily as input
	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR6; 	//pull-up/down off,wasts current
//handle the different I/O cases:
	if (gfPWM1 < 0)  {
		GPIOA->PUPDR |= GPIO_PUPDR_PUPDR6_1;	//pull-down
		TIM3->CCER &= ~TIM_CCER_CC1E;			//disable output handling
		return;
	}
	if (gfPWM1 == 0)  {
		GPIOA->MODER |= GPIO_MODER_MODER6_0 ;	// Set PortA6 as output
												//OTYPER as already config.
		GPIOA->BSRR = GPIO_BSRR_BR_6;			// permanent LOW
		TIM3->CCER &= ~TIM_CCER_CC1E;			//disable output handling
		return;
	}
	if (gfPWM1 >= 2000)  {
		gfPWM1 = 2000;
		GPIOA->MODER |= GPIO_MODER_MODER6_0 ;	// Set PortA6 as output
												//OTYPER as already config.
		GPIOA->BSRR = GPIO_BSRR_BS_6;			// permanent HIGH
		TIM3->CCER &= ~TIM_CCER_CC1E;			//disable output handling
		return;
	}
	//set PA6 in alternate function mode
	GPIOA->MODER |= GPIO_MODER_MODER6_1;		// Set PortA6 in AF mode
	GPIOA->AFR[0] &= 0xF0FFFFFF;				//TIM3,CH1 is AF1
	GPIOA->AFR[0] |= 0x01000000;
	TIM3->CCER |= TIM_CCER_CC1E;				//enable output handling
	TIM3->CCR1 = (uint32_t)gfPWM1;				//H period ca.11 bit resolution
}


void op_PWM2(void)  // ******************************************************
// ( pwm -- )
//start peripheral TIMER3, channel 2 to work on PA7 as output
// *************************************************************************
{
	if (gfExecUser) checkBackNum();

	gfPWM2 = (int32_t)*gfSP;
	gfSP --;

	if ((TIM3->CR1 & TIM_CR1_CEN) == 0)  {	//timer	emergency activation
		gfFreq = 24;
        subFREQ();
	}
	subPWM2();
}


void subPWM2(void)  // *****************************************************
// ( pwm -- )
//start peripheral TIMER3, channel 2 to work on PA7 as output
// *************************************************************************
{
	GPIOA->MODER &= ~(GPIO_MODER_MODER7);	//temporarily as input
	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR7; 	//pull-up/down off,wasts current
//handle the different I/O cases:
	if (gfPWM2 < 0)  {
		GPIOA->PUPDR |= GPIO_PUPDR_PUPDR7_1;	//pull-down
		TIM3->CCER &= ~TIM_CCER_CC2E;			//disable output handling
		return;
	}
	if (gfPWM2 == 0)  {
		GPIOA->MODER |= GPIO_MODER_MODER7_0 ;	// Set PortA7 as output
		GPIOA->BSRR = GPIO_BSRR_BR_7;			// permanent LOW
		TIM3->CCER &= ~TIM_CCER_CC2E;			//disable output handling
		return;
	}
	if (gfPWM2 >= 2000)  {
		gfPWM2 = 2000;
		GPIOA->MODER |= GPIO_MODER_MODER7_0 ;	// Set PortA7 as output
		GPIOA->BSRR = GPIO_BSRR_BS_7;			// permanent HIGH
		TIM3->CCER &= ~TIM_CCER_CC2E;			//disable output handling
		return;
	}
	//set PA7 in alternate function mode
	GPIOA->MODER |= GPIO_MODER_MODER7_1;		// Set PortA7 in AF mode
	GPIOA->AFR[0] &= 0x0FFFFFFF;				//TIM3,CH2 is AF1
	GPIOA->AFR[0] |= 0x10000000;
	TIM3->CCER |= TIM_CCER_CC2E;				//enable output	handling
	TIM3->CCR2 = (uint32_t)gfPWM2;				//H period ca. 11 bit resolution
}

#ifndef USARTIO
void op_MOT(void)  // ******************************************************
// ( StepSpeed -- )
//starts background process, controlled by "system"user VARiable NSTP
//2-Phase pulse scheme, must be expanded externally to 4 phases and amplified
//2 drive ports PA0 and PA1
//control of motor power must be handled externally
//StepSpeed=0:stepper off, negative value: rotation counter clockwise
//is counted down with 1ms clock (Arm SsyTick)
//gfUserHeader[4].toCompile is the actual number of steps to be performed.
//							is reset when number of steps were performed
//NSTP= 0:rotation stops, else number of steps to be performed
//NSTP= -2-->permanent rotation in stepper mode
//NSTP= -1:NORMAL (commutated) MOTOR drive  (steady H Bridge, no steps)
//then: 
// if (stepSpeed >= 0) rotation clockwise, PWM1 is started with value of speed
// else rotation counter clockwise, PWM1 is started with abs(value) of speed
// *************************************************************************
{
	if (gfExecUser) checkBackNum();

	if (gfUserHeader[4].toCompile == 0xFFFFFFFF)  { //signed int -1
		gfStepSpeed = 0;					//neutralize background process
		GPIOA->MODER &= ~(GPIO_MODER_MODER0 | GPIO_MODER_MODER1);
		GPIOA->MODER |= (GPIO_MODER_MODER0_0 | GPIO_MODER_MODER1_0);	//output
		if ((int32_t)*gfSP > 0)  {			//drive normal motor clockwise
			GPIOA->BSRR = GPIO_BSRR_BR_0;
			GPIOA->BSRR = GPIO_BSRR_BS_1;
		}
		else if ((int32_t)*gfSP < 0)  {			//drive motor counterClockwise
			*gfSP = -(*gfSP);
			GPIOA->BSRR =  GPIO_BSRR_BS_0;
			GPIOA->BSRR =  GPIO_BSRR_BR_1;
		}
		else  {	//==0							//includes set PWM1->0
			GPIOA->BSRR =  GPIO_BSRR_BR_0;		//stop motor
			GPIOA->BSRR =  GPIO_BSRR_BR_1;
		}
		op_PWM1();	//start PWM @PA6 with value of TOS 
					//PWM ratio limited to max 2000
		return;
	}
	//else
	if (gfStepSpeed == 0)  {			//old value
		gfStepPhase = 0; 
		gfSTEPCOUNT = 0;				//restart new phase
//override user GPIO setup
		GPIOA->MODER &= ~(GPIO_MODER_MODER0 | GPIO_MODER_MODER1);
		GPIOA->MODER |= GPIO_MODER_MODER0_0 | GPIO_MODER_MODER1_0; //output
	}
	gfStepSpeed = *gfSP;
	gfSP--;
	if (gfStepSpeed == 0)  {			//new value
		GPIOA->BSRR = GPIO_BSRR_BR_0;	//switch motor power off externally!
		GPIOA->BSRR = GPIO_BSRR_BR_1;
		gfUserHeader[4].toCompile = 0;
	}
	gfSTEPCOUNT = 0;					//restart new phase, trigger subMOT
	subMOT();
}


void subMOT(void)  // ******************************************************
//handles next stepper step in background process
//comment see above
// *************************************************************************
{
int32_t direction, nsteps;

//drive stepper motor
	nsteps = gfUserHeader[4].toCompile;			//user access: NSTP
	if ((nsteps == 0)||(nsteps == 0xFFFFFFFF)) return;//final or DC motor mode

	direction = gfStepSpeed;
	if (direction >= 0)  {	//run clockwise
		gfSTEPCOUNT = gfStepSpeed;
		switch (gfStepPhase)  {
		case 0:
			GPIOA->BSRR = GPIO_BSRR_BR_0;
			GPIOA->BSRR = GPIO_BSRR_BR_1;		//repetition essential
			break;
		case 1:
			GPIOA->BSRR = GPIO_BSRR_BS_1;
			break;
		case 2:
			GPIOA->BSRR = GPIO_BSRR_BS_0;
			break;
		case 3:
			GPIOA->BSRR = GPIO_BSRR_BR_1;
			break;
		}
	}
	else  {										//run counter clockwise
		gfSTEPCOUNT = -gfStepSpeed;
		switch (gfStepPhase)  {
		case 0:
			GPIOA->BSRR = GPIO_BSRR_BR_0;		//repetition essential
			GPIOA->BSRR = GPIO_BSRR_BR_1;
			break;
		case 1:
			GPIOA->BSRR = GPIO_BSRR_BS_0;
			break;
		case 2:
			GPIOA->BSRR = GPIO_BSRR_BS_1;
			break;
		case 3:
			GPIOA->BSRR = GPIO_BSRR_BR_0;
			break;
		}
	}
	gfStepPhase ++;
	if (gfStepPhase >= 4)  gfStepPhase = 0;
	if (nsteps != 0xFFFFFFFE) gfUserHeader[4].toCompile = nsteps - 1;
				//update countdown , nsteps=0xFFFFFFFE=-2: permanent rotation
}
#endif

// peripheral ops ##########################################################

uint32_t checkValidIO(void)  // ********************************************
{
uint32_t pinNo;

    pinNo = *gfSP;
    gfSP--;

	if (pinNo > 0x17)  {						//PB8 is grounded by HW
		if ((pinNo < 0xA0) || (pinNo > 0xB7)) gfErrors |= ERR_NUMVAL;
		pinNo &= 0x0000001F;			//PortB = bit4 set, range 0...0x18=24
	}
#ifdef USARTIO
	if (pinNo <= 15)  {
		if ((pinNo != 11) && ((pinNo == 2) || (pinNo >= 9)))  {
				//PA2=STLINK_TX, PA9,10 is USART1
				//PA12 is jumper, PA15=STLINK_RX, PA13,14<-STLink
			gfErrors |= ERR_NUMVAL;
		}
	}
	else  {
		if ((pinNo == 0x12) || (pinNo == 0x13)) {	
			// PB0 is jumper,PB2 not implemented, PB3 is UserLed
			gfErrors |= ERR_NUMVAL;
		}
	}
#endif
	return pinNo;
}

void checkPWM(uint32_t pinNo)  // ******************************************
// *************************************************************************
{
	if (pinNo == 6)  {
		if (gfPWM1 != -1)  {
			TIM3->CCER &= ~TIM_CCER_CC1E;		//disable output handling
			gfPWM1 = -1;						//PWM OFF
		}
	}
		if (pinNo == 7)  {
			if (gfPWM2 != -1)  {
			TIM3->CCER &= ~TIM_CCER_CC2E;		//disable output handling
			gfPWM2 = -1;						//PWM OFF
		}
	}
}


void op_PINOH(void)  // *****************************************************
// (Port/Pin -- )
// sets GPIO to output HIGH in Push/Pull mode
// parameter: pin number: if Number Base= Hex:A0,A1,A4..A7,AF,B0,B3,B4,.,B8
// else pin numbers must be entered with leading 0x, e.g. 0xA0,0xAA,0xB7
// or as corresponding decimal number, e.g. 0xA0=160, 0xAA=170,0x B0=176
// 		more simple:subtract 160 from decimal value:e.g. PA15=15,PB0=16,PB8=24
// specific pin numbers are reserved for system or not implemented
// *************************************************************************
{
uint32_t pinNo, pos, mask;

//simplified test for PortA and PortB
	pinNo = checkValidIO();
	if (gfErrors) return;

	if (pinNo <= 15)  {
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo >= 13))  {
								//PA2=USART_TX,PA15=STLINK_RX, PA13,14<-STLink
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		checkPWM(pinNo);

		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOA->PUPDR &= mask;				//pull-up/down off, wasts current
		GPIOA->MODER &= mask;
		GPIOA->MODER |= 1 << pos;
		GPIOA->BSRR = 1 << pinNo;
	}
	else  {
		pinNo -= 0x10;
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo == 3)) {	
							//PB2 not implemented,PB3 is UserLED
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOB->PUPDR &= mask;				//pull-up/down off, wasts current
		GPIOB->MODER &= mask;
		GPIOB->MODER |= 1 << pos;
		GPIOB->BSRR = 1 << pinNo;
	}
}


void op_PINOL(void)  // ****************************************************
// (Port/Pin -- )
// sets GPIO to output LOW
// comment about pinNo see op_PINOH
// *************************************************************************
{
uint32_t pinNo, pos, mask;

//simplified test for PortA and PortB
	pinNo = checkValidIO();
	if (gfErrors) return;

	if (pinNo <= 15)  {
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo >= 13))  {
								//PA2=USART_TX,PA15=STLINK_RX, PA13,14<-STLink
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		checkPWM(pinNo);
		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOA->PUPDR &= mask;				//pull-up/down off, wasts current
		GPIOA->MODER &= mask;
		GPIOA->MODER |= 1 << pos;
		GPIOA->BSRR =  1 << (pinNo + 16);
	}
	else  {
		pinNo -= 0x10;
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo == 3)) {	
							//PB2 not implemented,PB3 is UserLED
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOB->PUPDR &= mask;				//pull-up/down off, wasts current		
		GPIOB->MODER &= mask;
		GPIOB->MODER |= 1 << pos;
		GPIOB->BSRR = 1 << (pinNo + 16);
	}
}


void op_PINIPU(void)  // ****************************************************
// (Port/Pin -- )
// configures GPIO as input with internal pull-up resistor
// comment about pinNo see op_PINOH
// *************************************************************************
{
uint32_t pinNo, pos, mask;

//simplified test for PortA and PortB
	pinNo = checkValidIO();
	if (gfErrors) return;

	if (pinNo <= 15)  {
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo >= 13))  {
								//PA2=USART_TX,PA15=STLINK_RX, PA13,14<-STLink
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		checkPWM(pinNo);
		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOA->MODER &= mask;					//configured as input
		GPIOA->PUPDR &= mask;					//first HiZ
		GPIOA->PUPDR |= 1 << pos;				//now pull-up
	}
	else  {
		pinNo -= 0x10;
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo == 3)) {	
							//PB2 not implemented,PB3 is UserLE
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		pos = pinNo << 1;
		mask = ~(3 << pos);
        GPIOB->MODER &= mask;
        GPIOB->PUPDR &= mask;					//first HiZ
        GPIOB->PUPDR |= 1 << pos;				//now pull-up
    }
}


void op_PINIPD(void)  // ***********************************************
// (Port/Pin -- )
// configures GPIO as input with internal pull-down resistor
// comment about pinNo see op_PINOH
// *************************************************************************
{
uint32_t pinNo, pos, mask;

//simplified test for PortA and PortB
	pinNo = checkValidIO();
	if (gfErrors) return;

	if (pinNo <= 15)  {
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo >= 13))  {
								//PA2=USART_TX,PA15=STLINK_RX, PA13,14<-STLink
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		checkPWM(pinNo);
		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOA->MODER &= mask;					//configured as input
		GPIOA->PUPDR &= mask;					//first HiZ
		GPIOA->PUPDR |= 2 << pos;				//now pull-down
	}
	else  {
		pinNo -= 0x10;
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo == 3)) {	
							//PB2 not implemented,PB3 is UserLED
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		pos = pinNo << 1;
		mask = ~(3 << pos);
        GPIOB->MODER &= mask;
        GPIOB->PUPDR &= mask;					//first HiZ
        GPIOB->PUPDR |= 2 << pos;				//now pull-down
    }
}


void op_PINIZ(void)  // ****************************************************
// (Port/Pin -- )
// configures GPIO as HI-Z input
// comment about pinNo see op_PINOH
// *************************************************************************
{
uint32_t pinNo, pos, mask;

//simplified test for PortA and PortB
	pinNo = checkValidIO();
	if (gfErrors) return;

	if (pinNo <= 15)  {
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo >= 13))  {
								//PA2=USART_TX,PA15=STLINK_RX, PA13,14<-STLink
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		checkPWM(pinNo);

		pos = pinNo << 1;
		mask = ~(3 << pos);
		GPIOA->MODER &= mask;					//configured as input
		GPIOA->PUPDR &= mask;					//HiZ
	}
	else  {
		pinNo -= 0x10;
#ifndef USARTIO
		if ((pinNo == 2) || (pinNo == 3)) {	
							//PB2 not implemented,PB3 is UserLED
			gfErrors |= ERR_NUMVAL;
			return;
		}
#endif
		pos = pinNo << 1;
		mask = ~(3 << pos);
        GPIOB->MODER &= mask;					//configured as input
        GPIOB->PUPDR &= mask;					//HiZ
    }
}


void op_READDIG(void)  // *************************************************
// (port -- digitalLevel)
// returns digital level of addressed GPIO
// pin is always read, 
// no matter if port the is configured as digital input or reserved
// comment about pinNo see op_PINOH
// *************************************************************************
{
uint32_t pinNo, level;

	pinNo = checkValidIO();
	if (gfErrors) return;

	if (pinNo <= 15) level = GPIOA->IDR & (1 << pinNo);
	else  {
		pinNo &= 0x0000000F;		//extract portPin number
		level = GPIOB->IDR & (1 << pinNo);
	}
//even if pin is not implemented by hardware
	gfSP++;
	if (level)  *gfSP = 1;
	else  *gfSP = 0;
}


void op_READADC(void)  // **************************************************
//(PortApin_number -- analogLevel)
//returns analog level of addressed port. Initalizes pin for ADC automatically
//only PA0,PA1,PA4,PA5,PA6,PA7 are supported
//parameter: formal pin number A0,A1, A4...A7 or simply as number 0,1,4,5,6,7
// *************************************************************************
{
uint32_t port, mask, channel;
const uint8_t initStr[] = "  (ADC init) ";
const uint8_t offStr[] = "  (ADC off) ";

	port = *gfSP;
	if (port == 0xFFFFFFFF)  {		//i.e. signed -1
		gfSP--;
		disableADC();
		sendString(offStr);
		return;
	}
	if ((port > 7) && ((port < 0xA0) || (port  > 0xA7)))  {
			 gfErrors |= ERR_NUMVAL;
			return;
	}

    if ((RCC->APB2ENR & RCC_APB2ENR_ADCEN) == 0)  {
       initADC();
       sendString(initStr);
    }
    ADC1->ISR = 0x000007FF;			//clear relevant all bits by writing 1

    port &= 0x0000000F;
    mask = 3 << (port << 1);

    switch (port)  {
    case 0:
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 0;
        break;
    case 1:
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 1;
        break;
    case 2:							//PA2 is USART2 Tx
		gfErrors |= ERR_NUMVAL;
		return;
    case 3:
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 3;
        break;
    case 4:
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 4;
        break;
    case 5:		
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 5;
        break;
    case 6:
		if (gfPWM1 != -1)  {
			TIM3->CCER &= ~TIM_CCER_CC1E; //disable output handling
			gfPWM1 = -1;
		}
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 6;
        break;
    case 7:
		if (gfPWM2 != -1)  {
			TIM3->CCER &= ~TIM_CCER_CC2E; //disable output handlin
			gfPWM2 = -1;
		}
        GPIOA->MODER |= mask;		//analog input
        GPIOA->PUPDR &= ~mask;		//HiZ
        channel = 7;
    }
    ADC1->CHSELR = (1 << channel);

//result is improved by multiple ADC sampling
    mask = 0;
    for (port = 0; port < 4; port++) {	//make average of 8 samples
        ADC1->CR |= ADC_CR_ADSTART;
        while ((ADC1->ISR & ADC_ISR_EOC) == 0) ;
		mask += (uint32_t)ADC1->DR;
		wait5us();
    }
    *gfSP = mask >> 2;		//div/4
	if (mask & 0x00000002)  (*gfSP)++;	//round up if remainder(div/4) > 1
/*
	//alternative if fast result is needed:
	ADC1->CR |= ADC_CR_ADSTART;
	while ((ADC1->ISR & ADC_ISR_EOC) == 0) ;
	*gfSP= (uint32_t)ADC1->DR;
*/
}


void initADC(void)  // ***************************************************
// ( -- )
// initalizes ADC in general
// once initialized, it will be active during the session.
// takes some extra current
// *************************************************************************
{
	RCC->APB2RSTR &= ~(RCC_APB2RSTR_ADCRST);
    RCC->APB2ENR |= RCC_APB2ENR_ADCEN;
//AVR voltage regulator starting time is regular 20us
    wait5us();							//as a precaution
    wait5us();
    wait5us();
    wait5us();
    ADC1->CFGR1 = 0;					//right aligned, 12bit resolution
    ADC1->CFGR2 = ADC_CFGR2_CKMODE_1;//CKMODE=10,clock source=APB2_CLK/4=12MHz
	ADC1->SMPR = 2;					//sampling time about 13.5 clock cycles
    ADC->CCR = 0;						//no special ADC channels active
    ADC1->CR |= ADC_CR_ADEN;
    while ((ADC1->ISR & ADC_ISR_ADRDY) == 0) ;
	wait5us();
}


void disableADC(void)  // ***************************************************
// ( -- )
// initalizes ADC in general
// once initialized, it will be active during the session.
// takes some extra current
// *************************************************************************
{
	ADC1->ISR |= ADC_ISR_ADRDY;			//clear by writing 1 
	ADC1->CR |= ADC_CR_ADDIS;			//set ADDIS bit
	while ((ADC1->CR & ADC_CR_ADEN) != 0);
    RCC->APB2ENR &= ~(RCC_APB2ENR_ADCEN);
	RCC->APB2RSTR |= RCC_APB2RSTR_ADCRST;
}


void op_INITSPI(void)  // ***************************************************
// (Mode(0,1,2,3)  Bits(4..32) Clock -- )  //3 stack parameter !!
// SPI is always Master
// SPI uses PA11(SCK),PB4(MISO),PB5(MOSI) 
// /CS must be handled by user as specigically needed
// *************************************************************************
{
    gfSpiClock = *gfSP;
    gfSP--;
    gfSpiBits = *gfSP;
    gfSP--;
    gfSpiMode = *gfSP;
    gfSP--;
    subInitSpi();
}

void subInitSpi(void)  // **************************************************
//mode0:data sampled on rising clk edge and shifted out at the falling edge
//		clock idle low
//mode1::data sampled on falling clk edge and shifted out at the rising edge
//		clock idle low
//mode2:data sampled on falling clk edge and shifted out at the rising edge
//		clock idle high
//mode3:data sampled on rising clk edge and shifted out at the falling edge
//		clock idle high
// *************************************************************************
{
	if ((gfSpiMode>3)||(gfSpiBits<4)||(gfSpiBits>32)||(gfSpiClock>0xFFFF)) {
		gfErrors |= ERR_NUMVAL;
		return;
	}

//PA11 is SPI1_SCK   configure as output push/pull (PB3 is User LED)
//PB4 is SPI1_MISO  configure as input pull down
//PB5 is SPI1-MOSI, configure as output push/pull

	GPIOA->MODER &= ~(GPIO_MODER_MODER11);			//temporarily PA11 input
	GPIOA->MODER|=GPIO_MODER_MODER11_0;				//PA11 push/pull out
	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR11;			//pull-down wasts current

	GPIOB->MODER &= ~(GPIO_MODER_MODER4|GPIO_MODER_MODER5);//temporarily input
	GPIOB->MODER |= GPIO_MODER_MODER5_0;			//PB5 MOSI push/pull out
	GPIOB->PUPDR &= ~(GPIO_PUPDR_PUPDR4|GPIO_PUPDR_PUPDR4);//PB5 wasts current
	GPIOB->PUPDR |= GPIO_PUPDR_PUPDR4_1;			//PB4=MISO now pull-down

	if (gfSpiMode >= 2) GPIOA->BSRR = GPIO_BSRR_BS_11;//preset idle SCLK High
	else  GPIOA->BSRR = GPIO_BSRR_BR_11;			//idle SCLK Low
	send0X32(GPIOA->MODER);
	sendTB(' ');
}


void op_SPI(void)  // ******************************************************
//(sendWord(or Byte) --	recvWord(or	Byte))
//uses default parameters or user parameters if project	is loaded from Flash,
//else these must be initialised before with INITSPI()
// /CS must	be handled completely by the user, how ever
// *************************************************************************
{
uint32_t spiTX,	spiRX;
int32_t	k, loops;

	if (((GPIOA->MODER & 0x00C00000) != 0x00400000) ||
			((GPIOB->MODER & 0x00000C00) != 0x00000400)) subInitSpi();
											//i.e.PA11,PB5=output,PB4=input
				//if this does not fit->emergency init with actual parameters

//for precaution, usually not necessary:
	if (gfSpiMode >= 2) GPIOA->BSRR = GPIO_BSRR_BS_11;	//idle SCLK High
	else  GPIOA->BSRR = GPIO_BSRR_BR_11;					//idle SCLK Low

	spiTX = *gfSP;
	spiRX = 0;
	k = gfSpiBits;

	do  {
		k--;
		if (spiTX & (1<< k)) GPIOB->BSRR = GPIO_BSRR_BS_5;//MOSI high
		else GPIOB->BSRR = GPIO_BSRR_BR_5;				//MOSI Low

		if ((gfSpiMode == 0) || (gfSpiMode == 3))  {
			GPIOA->BSRR = GPIO_BSRR_BR_11;					//SCLK	Low
		}
		else GPIOA->BSRR = GPIO_BSRR_BS_11;					//SCLK	High

		loops = gfSpiClock;
		while (loops) {
			loops--;
		}

		spiRX = spiRX << 1;
		spiRX |= (GPIOB->IDR >> 4) & 1;						//PB4-->bit0

		if ((gfSpiMode == 0) || (gfSpiMode == 3))  {
			GPIOA->BSRR = GPIO_BSRR_BS_11;					//SCLK	High
		}
		else GPIOA->BSRR = GPIO_BSRR_BR_11;					//SCLK	Low


		loops = gfSpiClock;
		while (loops) {
			loops--;
		}
	}  while (k);

	if (gfSpiMode >= 2) GPIOA->BSRR = GPIO_BSRR_BS_11;	//idle SCLK High
	else  GPIOA->BSRR = GPIO_BSRR_BR_11;					//idle SCLK Low
	* gfSP = spiRX;
}

#ifndef USARTIO
void op_REGWRITE(void)  // *************************************************
// (value  peripheral_register_address -- )
// writes "value" the addressed peripheral register
// register address: see Reference Manual and/or STM32G431xx.h
// *************************************************************************
{
uint32_t* RegAddr;

    if ((*gfSP < 0x40000000)||(*gfSP > 0x50060C00)) {
										//BOT&TOP see RefMan
        gfErrors |= ERR_NUMVAL;
        return;
    }
    RegAddr = (uint32_t*)*gfSP;
    *RegAddr = *(gfSP-1);
    gfSP -= 2;
}


void op_REGREAD(void)  // **************************************************
// (peripheral_register_address -- value)
// returns the content of the addressed peripheral register
// register address: see Reference Manual and/or STM32G431xx.h
// *************************************************************************
{
uint32_t* RegAddr;

    if ((*gfSP < 0x40000000)||(*gfSP > 0x50060C00)) {
										//BOTTOM&TOP see RefMan

        gfErrors |= ERR_NUMVAL;
        return;
    }
    RegAddr = (uint32_t*)*gfSP;
    *gfSP = *RegAddr;
}
#endif

// structuring ops  ########################################################

void op_IF(void)  // *******************************************************
//( -- addr )
//compiles doIF
//starts compilation of a IF ... ELSE ... THEN conditional
//At compile time,it compiles doIF, provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOIF);
	prepareElse();
}


void op_CASE(void)  // *******************************************************
//( -- addr )
//compiles doIF
//starts compilation of a CASE ... ELSE ... THEN conditional
//At compile time,it compiles doCASE, provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOCASE);
	prepareElse();
}


void op_LTCASE(void)  // *************************************************
//( -- addr )
//compiles do<CASE
//starts compilation of a <CASE ... ELSE ... THEN conditional
//At compile time,it compiles do<CASE,provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOLTCAS);
	prepareElse();
}


void op_GTCASE(void)  // *************************************************
//( -- addr )
//compiles do>CASE
//starts compilation of a >CASE ... ELSE ... THEN conditional
//At compile time,it compiles do<CASE,provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	compileTCode(IDX_DOGTCAS);
	prepareElse();
}


void prepareElse(void)  // *************************************************
//( -- addr )
//compiles common code of IF, CASE, <CASE
//At compile time,it compiles it provides a placeholder for jump address
//and sends its code address via stack to ELSE or THEN
// *************************************************************************
{
	gfSP++;
	*gfSP = gfCodeIndex;        //keep in mind for ELSE or THEN
	compileTCode(0xFF);
    compileTCode(0xFF);			//placeholder, will be filled by ELSE or THEN
}


void op_ELSE(void)  // *****************************************************
//( addr -- addr)
//compiles doELSE
//compiles middle part of a IF (or CASE, <CASE)... ELSE ... THEN conditional
//At compile time, it compiles doELSE ,its code address into the placeholder
//behind doIF and provides a placeholder for jump address.
//The stack entry with the code address provided by IF is
//exchanged with an entry that contains the code address of doELSE.
// *************************************************************************
{
uint32_t  temp, target;

	compileTCode(IDX_DOELSE);
	temp = gfCodeIndex;		            //keep in mind for THEN
	compileTCode(0xFF);
    compileTCode(0xFF);			//placeholder, will be filled by THEN
    target = (uint32_t)(gfTCode + gfCodeIndex -1) & 0x0000FFFF;
	gfTCode[*gfSP] = (uint8_t)(target >> 8);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
							//write jump target into compilation of IF
//calculated 1 byte less because pointer is incremented in executeTCode()
	*gfSP = temp;
}


void op_THEN(void)  // ****************************************************
//( addr -- )
//compiles termination of a IF ... ELSE ... THEN conditional
//Essentially it compiles the terminating code address
//into the placeholder behind doIF (or doELSE if present).
// *************************************************************************
{
uint32_t target;

    target = (uint32_t)(gfTCode + gfCodeIndex -1);
    target &= 0x0000FFFF;
	gfTCode[*gfSP] = (uint8_t)(target >> 8);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
							//write jump target into compilation of IF or ELSE
							//word pointer index offset is bytes*4
//calculated 1 word less because pointer is incremented in executeTCode()
	gfSP--;
}


void op_DO(void)  // *******************************************************
//( -- addr )
//compiles doDO
//starts compilation of a DO ... WHILE or DO ... AGAIN loop.
//Essentially it provides a placeholder for the BREAK code address
//and sends its code address via stack to WHILE or AGAIN
// *************************************************************************
{
    compileTCode(IDX_DODO);//at runtime DODO puts BREAK adddress on Return Stack
	gfSP++;
	*gfSP = gfCodeIndex;        //sends loopback address to "WHILE" or "AGAIN"
	compileTCode(0xFF);
    compileTCode(0xFF);			//placeholder for BREAK adddress
                                //will be filled by WHILE or AGAIN
}


void op_UNTIL(void)  // ****************************************************
//( addr -- )
//compiles doUNTIL
//finishes compilation of a DO ... WHILE loop
//Essentially it compiles its code address for BREAK behind doDO
//and compiles the loopback code adress provided by DO behind doUNTIL.
// *************************************************************************
{
	compileTCode(IDX_DOUNTIL);
	prepareBack();
}


void op_WHILE(void)  // ****************************************************
//( addr -- )
//compiles doWHILE
//finishes compilation of a DO ... WHILE loop
//Essentially it compiles its code address for BREAK behind doDO
//and compiles the loopback code adress provided by DO behind doWHILE.
// *************************************************************************
{
	compileTCode(IDX_DOWHILE);
	prepareBack();
}


void op_AGAIN(void)  // ***************************************************
//(   -- )
//compiles doAGAIN
//finishes compilation of a DO ... AGAIN loop
//Essentially it inserts its code address for BREAK compilation behind doDO
//and compiles the loopback code adress provided by DO behind doAGAIN.
// *************************************************************************
{
	compileTCode(IDX_DOAGAIN);
	prepareBack();
}


void prepareBack(void)  // ********************************************************
// *************************************************************************
{
uint32_t target;

	target = (uint32_t)(gfTCode + *gfSP + 1);
	compileTCode((uint8_t)((target >> 8) & 0x000000FF));
	compileTCode((uint8_t)(target & 0x000000FF));//loopback address sent by DO
	target =  (uint32_t)(gfTCode + gfCodeIndex -1);
	gfTCode[*gfSP] = (uint8_t)((target >> 8) & 0x000000FF);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
//gfCodeIndex was incremented by "compileTCode()" ie. points to RPDEC then.
//must be decremented because gfIP is incremented when doBREAK etc. is left
	compileTCode(IDX_RPDEC);		//compile ReturnStach corrections handler
	gfSP--;
}


void op_FOR(void)  // ******************************************************
//( -- addr )
//compiles doFOR
//starts compilation of a FOR ... LOOP loop
//The runtime behaviour is very similar to "C" programming:
//doFOR expects 3 DataStack entries: startIndex, endIndex, Increment
//Increment can be positive or negative and can be greater than 1.
//For runtime details of the FOR...LOOP loop see doFOR and doLOOP below.
//Essentially FOR provides a placeholder for the BREAK code address
//and sends its code address via stack to LOOP.
// *************************************************************************
{
    compileTCode(IDX_DOFOR);//at runtime DOFOR puts BREAK adddress on Ret.Stack
	gfSP++;                     //sends the return address to "LOOP"
	*gfSP = gfCodeIndex;
	compileTCode(0xFF);
    compileTCode(0xFF);		//placeholder, will be filled by LOOP
}


void op_LOOP(void)  // *****************************************************
//( addr -- )
//compiles doLOOP
//finishes compilation of a FOR ... LOOP loop
//Essentially it inserts its code address for BREAK compilation behind doFOR
//and compiles the loopback code adress provided by FOR behind doLOOP.
// *************************************************************************
{
uint32_t target;

	compileTCode(IDX_DOLOOP);
	target = (uint32_t)(gfTCode + *gfSP + 1);
	compileTCode((uint8_t)((target >> 8) & 0x000000FF));
	compileTCode((uint8_t)(target & 0x000000FF));//compile jump back behind FOR
	target =  (uint32_t)(gfTCode + gfCodeIndex -1);
	gfTCode[*gfSP] = (uint8_t)((target >> 8) & 0x000000FF);
	gfTCode[*gfSP + 1] = (uint8_t)(target & 0x000000FF);
									//sends the CONTI & BREAK address to "FOR"
//gfCodeIndex was incremented by "compileTCode()" ie. points to RP4DEC then.
//must be decremented here because gfIP is incremented when doBREAK etc. is left
	compileTCode(IDX_RP4DEC);
	gfSP--;
}

void op_IDX(void)  // ******************************************************
//( -- index)
//at runtime= copies 2nd Return Stack entry = actual loop parameter on TOS
// *************************************************************************
{
    gfSP++;
    *gfSP = *(gfRP - 1);
}


void op_CONTI(void)  // ****************************************************
//( --- )
//initiates unconditional jump to behind doDO or doFOR (starts new loop)
//performs similar action as BREAK,
//but jumps to code behind (doDO+arg.) or (doFOR+arg.)
//More than one CONTI may be within a loop, each w. individual decision
//gfIP will be incremented when op_DOCONTI is left
// *************************************************************************
{
uint8_t *cptr;
uint32_t temp;

	temp = (*gfRP - 2) | 0x20000000;
	cptr = (uint8_t*)temp;
	temp = *cptr;
	if (temp == IDX_DOLOOP)  gfIP = cptr - 1;
	else  {
		gfIP = cptr;
		op_DOAGAIN();
	}
}


void op_BREAK(void)  // ****************************************************
//( --- )
//initiates unconditional jump to (value from Return Stack)
// *************************************************************************
{
	gfIP = (uint8_t*)*gfRP;
        //the BREAK IP is stored in *gfRP by doDO or doFOR
        //gfIP will be incremented in "executeTcode()" when op_BREAK is left
        //gfRP is reset by RP-1 (DO loop) or RP-4 (FOR loop)
}


void op_RETURN(void)  // ***************************************************
//( --- )
//leaves a user thread before end (equivalent to "C" return;)
//see op_SEMIS , too
// *************************************************************************
{
	gfIP = (uint8_t*)*gfRP;
	if (gfRP > gfReturnStack) {
//new 31Dec23:
		if (gfBackProcess == 2)  {
			if (gfNestLevel == gfBackNestLevel)  {
				gfBackNestLevel = 0;
				gfBackProcess = 3;
			}
		}
		gfRP--;				//else gfRP==gfReturnStack-->finished
		gfNestLevel --;
	}
	else  {
		gfIP =  0;		// signal termination to executeTcode()
		if (gfBackProcess == 2)  gfBackProcess = 3;
	}
}


// compiler ops  ###########################################################

void op_PRINTSTACK(void)  // ***********************************************
// simple debug function: displays all stack entries. TOS is rightmost
//for debug purposes exceptionally it can be compiled into User Threads
// *************************************************************************
{
uint32_t k;
const uint8_t UnderflowStr [] = "underflow! ";
const uint8_t emptyStr [] = " --- ";

    sendTB('[');
	if (gfSP == (gfDataStack + 3))  sendString(emptyStr);
    else if (gfSP < gfDataStack + 3)  {
        sendString(UnderflowStr);
        gfSP = gfDataStack; //re-init stack
        *gfSP=0;
        gfSP++;
        *gfSP = 0;          //four stack items as underflow reserve
        gfSP++;
        *gfSP = 0;
        gfSP++;
        *gfSP = 0;
    }
    else  {
        k = 4;
        while ((gfSP > gfDataStack + 3)&&(k <= (uint32_t)(gfSP-gfDataStack))) {
							//4 stack items as underflow reserve not displayed
            sendTB(SPACE);
            if (gfTempNumBase == 16)  send0X32(gfDataStack[k]);
            else  sendNS32(gfDataStack[k]);
			sendTB(SPACE);
            k++ ;
        }
    }
    sendTB(']');
    sendTB(SPACE);
}


void op_FLUSH()  // ********************************************************
// *************************************************************************
{
	gfSP = gfDataStack;
	*gfSP=0;
	gfSP++;
	*gfSP = 0;                          //four stack items as underflow reserve
	gfSP++;
	*gfSP = 0;
	gfSP++;
	*gfSP = 0;	
}



void op_OPS(void)  // ******************************************************
//display all kernel operators: their names and their token values
// *************************************************************************
{
uint32_t k, n, p;
const uint8_t KernelStr [] = "Kernel:";

    sendString(KernelStr);
    sendTB(CRR);
    k = 1;				//don't show {ERROR} dummy name
    n = 0;
	while (k < KERNELSIZE) {
        sendTB(SPACE);
        for (p=0;p<9;p++)  {
            if (gfKernelNames[k][p]) sendTB(gfKernelNames[k][p]);
            else break;
        }
        while(p < 7) {
            sendTB(SPACE);
            p++;
        }

        sendTB('#');
        sendX8((uint8_t)k);
        sendTB(SPACE);
        sendTB(SPACE);
        sendTB(SPACE);
		k++;
		n++;
		if (n==5) {
            sendTB(CRR);
			n = 0;
		}
	}
}


void op_USER(void)  // *****************************************************
//display all user token: their names and their token values
// *************************************************************************
{
uint32_t k, p, wrap;
const uint8_t VarconStr [] = "(VarCon=";
    k = 0;
    wrap = 0;
    sendTB(SPACE);
	while (k < gfNextNameCompi)  {
        if (k<10) sendTB(SPACE);
        sendNS32(k);
        sendTB(':');
        for (p=0;p<8;p++)  {
            if (gfUserHeader[k].name[p]) sendTB(gfUserHeader[k].name[p]);
            else break;
        }

		if (gfUserHeader[k].attributes & A_VARCON)  {
			sendString(VarconStr);
            if (gfTempNumBase == 16)  send0X32(gfUserHeader[k].toCompile);
            else sendNS32(gfUserHeader[k].toCompile);
            sendTB(SPACE);
            sendTB('@');
            send0X32(((uint32_t)&gfUserHeader[k].toCompile) & 0x0000FFFF);
			sendTB(0x29);     //')'
            wrap += 2;
        }
        sendTB(SPACE);
        sendTB(SPACE);
		k++;
		wrap++;
		if (wrap >= 6) {
            sendTB(CRR);
	        sendTB(SPACE);
			wrap = 0;
		}
	}
	if (wrap) sendTB(CRR);
}


void op_SEE(void)  // ******************************************************
//show the compiled body of a user thread
// *************************************************************************
{
uint32_t k, n, wrap, w, count, word, byte, address, index;
//uint8_t *letter;
const uint8_t KernelStr [] = "can't SEE Kernel Ops";
const uint8_t UserStr [] = "UserHeader_Struct=";
const uint8_t VarconStr [] = "VarCon:Value=";

	gfTokenStart = gfTibPos;

	scanNext();

	if (findUserToken() == 0) {	//updates "gfParsed"=index of "gfUserHeader"
		if (findKernelToken())  {
			sendString(KernelStr);
			return;
		}
		gfErrors |= ERR_STRINGTOKEN;
		return;
	}
	gfTokenStart = gfTibPos;
	sendString(UserStr);
	send0X32((uint32_t)gfUserHeader + gfParsed*16);	//16 bytes/struct
	sendTB(CRR);

	k = (gfUserHeader[gfParsed].toCompile - (uint32_t)gfTCode) & 0x00000FFF;

	wrap = 0;
	if ((gfUserHeader[gfParsed].attributes & A_VARCON) == 0) {
														//"normal" Function
		for (n = k;gfTCode[n] != IDX_SEMIS;n++) {
			wrap++;
			if (wrap >= 8) {
				sendTB(CRR);
				wrap = 0;
			}
			address = (uint32_t)(&(gfTCode[n])) & 0x0000FFFF;
			sendNX32(address);	//leading 0x2000.. and 0x suppressed
			sendTB(':');

			if (gfTCode[n] >= 0x80)  {					//new 18Jan25:
				index = gfTCode[n] & 0x7F;
				if (gfUserHeader[index].attributes & A_VARCON)  {
					index2String(index);
					sendTB(0x28);	 //'('
					sendTB('=');
					if (gfTempNumBase==16)  {
						send0X32(gfUserHeader[index].toCompile);
					}
					else  sendNS32(gfUserHeader[index].toCompile);
					sendTB(0x29);	 //')'
					sendTB(SPACE);
					sendTB(SPACE);
					continue;
				}
			}

			token2String(gfTCode[n]);
			if ((gfTCode[n] >= IDX_DOIF) && (gfTCode[n] <= IDX_DOLOOP)) {
				word = (uint32_t)(gfTCode[n+1] << 8) | (uint32_t)gfTCode[n+2];
				n += 2;
				sendTB(0x28);	 //'('
				sendNX32(word + 1);		//leading 0x suppressed
				//the compiled jump address is 1 less than effectively executed
				sendTB(0x29);	 //')'
				wrap++;
			}
			else if (gfTCode[n]==IDX_DOLIT8B) {
				n++;
				sendTB(0x28);	 //'('
				if (gfTempNumBase==16)  send0X32((uint32_t)gfTCode[n]);
				else  sendNS32((uint32_t)gfTCode[n]);
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOLIT) {
				word = (uint32_t)(gfTCode[n+1] << 24);
				word |= (uint32_t)(gfTCode[n+2] << 16);
				word |= (uint32_t)(gfTCode[n+3] << 8);
				word |= (uint32_t)gfTCode[n+4];
				n += 4;
				sendTB(0x28);	 //'('
				if (gfTempNumBase==16)  send0X32(word);
				else  sendNS32(word);
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOWVA) {
				n++;
				sendTB(0x28);	 //'('
				word  = (uint32_t)(gfTCode[n]);		//mod 01Jun21,bit7 not deleted
				index2String(word);
				word = (uint32_t)(&gfUserHeader[word].toCompile);
				sendTB('@');
				send0X32(word & 0x0000FFFF); //leading 0x2000.. suppressed
				sendTB(0x29);	 //')'
				wrap++;
			}

			else if (gfTCode[n]==IDX_DOSTR) {
				sendTB('"');
				n++;
				count = gfTCode[n];		//byte count, max.255!
				n++;
				while (count)  {
					byte  = gfTCode[n];
					n++;
					if ((byte >= 0x20) && (byte < 0x7F))  sendTB(byte);
					else  {
						sendTB(0x5C);        //Backslash
						sendNibble((byte >> 4) & 0x0000000F);
						sendNibble(byte & 0x0000000F);
					}
					count --;
				}
				n-- ;	//during while{} already increment by for{ preempted
				sendTB('"');
				wrap += 3;
			}
			sendTB(SPACE);
			sendTB(SPACE);
		}
		address = (uint32_t)(&(gfTCode[n])) & 0x0000FFFF;
		sendNX32(address);	//leading 0x2000.. and 0x suppressed
		sendTB(':');
		sendTB(';');
	}
	if (gfUserHeader[gfParsed].attributes & A_VARCON) {
		sendString(VarconStr);
		if (gfTempNumBase == 16)  send0X32(gfUserHeader[gfParsed].toCompile);
		else  sendS32(gfUserHeader[gfParsed].toCompile);
		sendTB(SPACE);
		sendTB('@');
		send0X32((uint32_t)(&gfUserHeader[gfParsed].toCompile) & 0x0000FFFF);
						//address always in hex, leading 0x2000.. suppressed
	}
	sendTB(CRR);
}


void token2String(uint32_t tokenval)   // **********************************
// *************************************************************************
{
uint32_t k, p;

	if (tokenval < KERNELSIZE) {        //Kernel token
        for (p=0;p<8;p++)  {
            if (gfKernelNames[tokenval][p]) sendTB(gfKernelNames[tokenval][p]);
            else break;
        }
	}
	else {                              //User token
		k = tokenval & 0x0000007F;
        for (p=0;p<8;p++)  {
            if (gfUserHeader[k].name[p]) sendTB(gfUserHeader[k].name[p]);
            else break;
        }
	}
}


void index2String(uint32_t index)   // *************************************
//reconstruct VARCON name from address of .toCompile  struct entry
// *************************************************************************
{
uint32_t p;
uint8_t letter;
	
	sendTB(0x27);	//= '
	for (p=0;p<8;p++)  {
		letter = gfUserHeader[index].name[p];
		if (letter) sendTB(letter);
		else break;
	}
	sendTB(0x27);
}


void op_MEM(void)  // *****************************************************
//(addr -- )
//send 128 byte entries starting from "addr" (16bit, w.o. leading 0x2000..)
//for debug purposes exceptionally it can be compiled into User Threads
//due to "little endian", native 32 bit entries (gfUserHeader[].toCompile, 
//and gfUserHeader[].Attributes) are displayed in reverse byte order
// *************************************************************************
{
uint8_t *address;
uint8_t value;
uint32_t k, wrap;
const uint8_t StartStr [] = "128 SRAM bytes starting from 0x";

	if (gfExecUser) checkBackNum();

    address = (uint8_t *)(0x20000000 | *gfSP);
	gfSP--;
    if ((uint32_t)address > 0x20007F80) {
        gfErrors |= ERR_NUMVAL;
        return;
    }
    sendTB(CRR);
    sendString(StartStr);
    sendX32((uint32_t)address);

    wrap = 15;
    for (k=0; k<128; k++)  {
        value = *(address + k);
        wrap++;
        if (wrap == 16) {
            sendTB(CRR);
            sendX32((uint32_t)(address + k));
            sendTB(':');
            wrap = 0;
        }
        sendX8(value);
        if ((wrap & 0x00000003) == 3) sendTB('|');
    }
    sendTB(CRR);
}


void op_STEP(void)  // ****************************************************
//( --- )
// activates single step execution, trigger next step with TAB key
// special case: key "ESC" terminates execution at once (leave endless loops)
//for debug purposes exceptionally it can be compiled into User Threads
// *************************************************************************
{
     gfBackProcess = 3;
}


void op_RUN(void)  // ****************************************************
//( --- )
//default state of background process, always active after system start 
//de-activates single step execution, but keeps the background process active
// special case: key "ESC" terminates execution at once (leave endless loops)
//for debug purposes exceptionally it can be compiled into User Threads
// *************************************************************************
{
    gfBackProcess = 1;
}


void op_NOBAK(void)  // ****************************************************
//( --- )
// de-activates background process. Token Code speed is approx 40% faster
//for speed control exceptionally it can be compiled into User Threads
// *************************************************************************
{
    gfBackProcess = 0;
}


void op_BAKOP(void)  // ****************************************************
//( --- )
// restores RUN mode when background process is OFF
// *************************************************************************
{
const uint8_t backStr [] = " ->BackOP ";
	if (gfBackProcess == 0)  {
		sendString(backStr);
		gfBackProcess = 1;
	}
}


void op_FORGET(void)  // **************************************************
//delete the argument user function and all later defined ones
//example:
// : XYZ .......
// FORGET XYZ
// *************************************************************************
{
uint32_t k;

	gfTokenStart = gfTibPos;
	scanNext();
	if (findUserToken() == 0) {
		gfErrors |= ERR_USERSTRING;
		return;
	}
	gfTokenStart = gfTibPos;

//update compiler pointers
#ifdef USARTIO
	if (gfParsed < 4)  {			//V1,2,3,4 cannot be deleted bugfix04Apr25
#else
	if (gfParsed < 5)  {			//NSTP, V1,2,3,4 cannot be deleted
#endif
		gfErrors |= ERR_CONTEXT;
		return;
	}
	gfNextNameCompi = gfParsed;
	gfUserHeader[gfParsed].name[0] = 0;
	gfCodeIndex = gfUserHeader[gfParsed].attributes & 0x00000FFF;
							//works with VARCON too doesn't modify gfCodeIndex

	if ((gfAuto - 1) >= gfParsed) gfAuto = 0;	//new 04Apr25
//as precaution, set the rest of gfUserHeader[] and gfTCode completely to zero
	initTop();
}


void op_ABORT()  // ********************************************************
//warm start. Re-initalizes runtime parameters and stacks,
//			  but keeps variable values and compiled functions
// *************************************************************************
{
uint32_t k;

//reconstruct project from data loaded from "popFlashProject()"
    gfComment = 0;
    gfCompiState = 0;
    gfExecUser = 0;
	gfBackNumValid = 0;
    gfNestLevel = 0;
	if (gfBackProcess)  gfBackProcess = 1;	//Bugfix 26Jan25
	gfTermCheck = 0;
	gfTempNumBase = gfNumBase;

    for (k = 0; k < MAXUSER; k++)  {		//clear thread of new token code
        if (gfUserHeader[k].name[0] == 0) break;
    }
    gfNextNameCompi = k;
	if ((gfAuto - 1) >= k) gfAuto = 0;		//new 04Apr25
    k = CODETOP;

    while (1) {
       if (gfTCode[k] == IDX_SEMIS) break;
       if (k == 0) break;
       k--;
    }
    if (k > 0) gfCodeIndex = k + 1;
    else gfCodeIndex = 0;

//as precaution, set the rest of gfUserHeader[] and gfTCode completely to zero
	initTop();

    gfRP = gfReturnStack;
	*gfRP = 0;	            //repeat as precaution, is essential when compiling
    gfIP = gfTCode;
	gfErrors = 0;
	op_FLUSH();
}


void initTop(void)  // *******************************************************
// ***************************************************************************
{
uint32_t k;
uint32_t *pointer;

    gfCodeStartIndex = gfCodeIndex;

    pointer = (uint32_t*)&(gfUserHeader[gfNextNameCompi]);
    while (pointer < (uint32_t*)&gfUserHeader[MAXUSER])  {
        *pointer = 0;
        pointer++;
    }
    for (k = gfCodeIndex; k < CODETOP; k++)  {
        gfTCode[k] = 0;
    }
}


void op_REPLACE(void)  // ***************************************************
// replace the token code of <target name> with that of <supplier name>
// Syntax:  REPLACE  <supplier name>  <target name>
// ( -- )  no stack parameters
// <supplier name> is always the first hit searched from TOP of userToken[]
// <target name> is always the first hit searched from TOP of userToken[]
// only token code between supplier and target is replaced
// If supplier and target have the same name,
// the search for target is continued down towards bottom of userToken[].
// original state is restored with "RESTORE"
//
// This PROCEDURE WORKS ONLY 
// on supplier, which is the first hit
// and target which is the first hit of search from TOP of userToken
// AND supplier is newer than target.
//
// If intended target is NEWER than supplier,a "retro" substitution is made
// as follows: compile a new UserToken with the code of old supplier.
//
// REPLACE inserts the supplier code address in gfUserHeader[target].toCompile
// but keeps the original gfTode index in gfUserHeader[target].attributes
// in 8bit token code format,a UserToken is the index in gfUserHeader[] |=0x80
// *************************************************************************
{
uint32_t   supIndex, targIndex;
uint32_t   supToken, targToken;

//parse <supplier name> from TIB
    supIndex = getToken();                 //index in gfUserHeader[] !!
    if (supIndex == 0xFFFFFFFF)  {
        gfErrors |= ERR_USERSTRING;
        return;
    }
    if (gfUserHeader[supIndex].attributes & A_VARCON)  {
        gfErrors |= ERR_CONTEXT;    //replacement of CONST or VAR makes no sense
        return;
    }
//parse <target name> from TIB
    targIndex =  getToken();                //index in gfUserHeader[] !!
    if (targIndex == 0xFFFFFFFF)  {
        gfErrors |= ERR_USERSTRING;
        return;
    }
    if (gfUserHeader[targIndex].attributes & A_VARCON)  {
        gfErrors |= ERR_CONTEXT; //replacement of CONST or VAR makes no sense
        return;
    }

    if (supIndex <= targIndex)  {
		gfErrors |= ERR_CONTEXT;
		return;
	}
	gfUserHeader[targIndex].toCompile = gfUserHeader[supIndex].toCompile;
}


void op_RESTORE(void)  // **************************************************
// Syntax:  RESTORE <original name>
// ( -- )  no stack parameters
// <original name> is always the first hit searched from TOP of userToken[].
// If this is "REPLACE"d, the threaded code is reset to the original, but only
// up in TCode[] to the start of the "native" code with "REPLACE"d token value.
// When it seems,that the code of the found token was not "REPLACE"d,the search
// is continued down to a possible second entry with same name in userToken[].
// If nothing has been "REPLACE"d, nothing gets changed
// The procedure is made possible as follows:
// REPLACE inserts the supplier code address in gfUserHeader[target].toCompile
// but keeps the original gfTode index in gfUserHeader[target].attributes
// in 8bit token code format,a UserToken is the index in gfUserHeader[] |=0x80
// *************************************************************************
{
 uint32_t  index, actualCode, origCode;

//parse <name of originalized token> from TIB
    index = getToken();                    //index in gfUserHeader[]
    if (index == 0xFFFFFFFF)  {
        gfErrors |= ERR_USERSTRING;
        return;
    }
    actualCode = gfUserHeader[index].toCompile;	//has been REPLACEd
//gfUserHeader[index].attributes contains another formulation of orig "toCompile"
    origCode = (uint32_t)((gfUserHeader[index].attributes & 0xFFF) + gfTCode);
												//=real code addresss in SRAM
//protect "native actualCode"
	if (actualCode < origCode)  {
		gfErrors |= ERR_CONTEXT;
		return;
	}
    gfUserHeader[index].toCompile = origCode; //restore in gfUserHeader[]
}


uint32_t getToken(void)   // ***********************************************
// *************************************************************************
{
int32_t k;
uint32_t n;
uint8_t refChar, userChar;

	gfTokenStart = gfTibPos;
	scanNext();
	gfTokenStart = gfTibPos;
	k = gfNextNameCompi - 1;	//k is index of last compiled USER Token
								//search DOWN from the LAST COMPILED

    while (k != -1)  {
        for (n=0; n < 16;n++) {
            userChar = gfUserHeader[k].name[n];
            refChar = gfTokenstr[n];
//transform into capital letters:
            if ((refChar >= 'a')&&(refChar <= 'z')) refChar -= 0x20;
            if (userChar != refChar)  break;
            if (userChar ==0) return k;//="gfUserHeader[]" index of found token
        }
        k--;
    }
	return 0xFFFFFFFF;				//if named token not found
}


void op_COLON(void)  // ****************************************************
//( --- )
// start the compilation of a user function.
// *************************************************************************
{
uint32_t  userSymbol;

	userSymbol = compileUserSymbol(0);
	if (userSymbol == 0) return;
	
//bugfix 30Dec23:
	if (userSymbol == 2)  gfUserHeader[gfNextNameCompi].name[0] = 0;	
							//compilation cancelled, mark end of Symbol Table
	else  {
//will be a "normal" token, no VARCON
		gfCodeStartIndex = gfCodeIndex;
		gfCompiState = 1;
	}
}


void op_SEMIS(void)  // *****************************************************
//( --- )
//terminates every compilation, i.e. must be the last thread token
// **************************************************************************
{
	if (gfCompiState != 0)  {						//during compilation
		compileTCode(IDX_SEMIS);
		gfCompiState = 0;
		gfNextNameCompi ++;
		gfUserHeader[gfNextNameCompi].name[0] = 0;	//mark end of Symbol Table
//update compiler pointers
		gfCodeStartIndex = gfCodeIndex;
	}
	else  {						//interpreting (runtime) state
		gfIP = (uint8_t*)*gfRP;
		if (gfRP > gfReturnStack) {
//new 31Dec23:
			if (gfBackProcess == 2)  {
				if (gfNestLevel == gfBackNestLevel)  {
					gfBackNestLevel = 0;
					gfBackProcess = 3;
				}
			}
            gfRP--;				//else gfRP==gfReturnStack-->finished
            gfNestLevel --;
		}
		else  {
			gfIP =  0;			// signal termination to executeTcode()
			if (gfBackProcess == 2)  gfBackProcess = 3;
		}
	}
	gfTempNumBase = gfNumBase;
}

void op_DOSTRING(void)  // *************************************************
//( --- )
//runtime operator, compiled by P"
//In ThreadCode memory it is followed by the bytes to be transmitted.
//differing from initial CForth versions, the string is "byte counted".
// *************************************************************************
{
	subDoString(1);
}

#ifdef USARTIO
void op_DOIOSTR(void)  // **************************************************
//( --- )
//runtime operator, compiled by TX"
//remark see op_DOSTRING
// *************************************************************************
{
	subDoString(0);
}
#endif

void subDoString(uint32_t out)  // *****************************************
// *************************************************************************
{
uint32_t n, count, word;

	gfIP++;
	count = *gfIP;
	gfIP++;
	n = 0;
	while (count)  {
		if (n == 0) {
			word = *gfIP;
			gfIP++;
		}
		if (out == 1) sendTB((word >> n) & 0xFF);
#ifdef USARTIO
		else sendIOB((word >> n) & 0xFF);
#endif
		n += 8;
		if (n > 24) n = 0;
		count --;
	}
	gfIP--;		//during while{} already increment by executeTCode() preempted
}


void op_DOLIT(void)  // ****************************************************
//( -- 32bitNumber)
//In the compiled thread, doLIT puts the content of the next ThreadCode entry
//(=number value) on TOS at runtime.
// *************************************************************************
{
uint32_t word;

	gfSP++;
    gfIP++;
    word = (uint32_t)(*gfIP << 24);
    gfIP++;
    word |= (uint32_t)(*gfIP << 16);
    gfIP++;
    word |= (uint32_t)(*gfIP << 8);
    gfIP++;
    word |= (uint32_t)*gfIP;
    *gfSP = word;
}


void op_DOLIT8B(void)  // ****************************************************
//( -- 8bitNumber)
//In the compiled thread, doLIT puts the content of the next ThreadCode entry
//(=number value) on TOS at runtime.
// *************************************************************************
{
	gfSP++;
    gfIP++;
    *gfSP = (uint32_t)*gfIP;
}

void op_DOWVA(void)  // *****************************************************
//(value -- )
//In the compiled thread,the "value" address of VARCON is comiled behind doWVA
//At runtime, "value" is written into this ".toCompile" field.
//different from G4xx version to save code length !!
//compileTCode() compiles only 8bit token, index is stored instead of address
// *************************************************************************
{
uint32_t index;
uint32_t* address;

	gfIP++;
	index = (uint32_t)(*gfIP);				//& 0x7F  deleted 01June21
	address = &gfUserHeader[index].toCompile;
	*address = *gfSP;
	gfSP--;
}


void op_DOIF(void)  // *****************************************************
//( flag --)
//compiled by IF
//takes a flag from TOS, if == 0, a jump is initiated:
//behind doELSE if present,
//or to code address compiled by THEN
//if != 0, execution is performed until doELSE.
//or continues linear (if no doELSE)
// *************************************************************************
{
uint32_t jump;

	if (*gfSP != 0)  gfIP += 2;
	else  toElse();
    gfSP--;
}


void op_DOCASE(void)  // *****************************************************
//( reference --)
//compiled by CASE
//compare reference with NOS
//if equal a jump is initiated behind doELSE if present,
//or to code address compiled by THEN
//if != 0, execution is performed until doELSE.
//or continues linear (if no doELSE)
//!! after this operation former NOS is TOS now for furter handling. 
//!! DROP if not needed
// *************************************************************************
{
uint32_t temp, jump;
	
    temp = *(gfSP-1);
	if (temp == *gfSP)  gfIP += 2;
	else  toElse();
    gfSP--;									//drop reference value
}


void op_DOLTCAS(void)  // ***************************************************
//( reference --)
//compiled by <CASE
//compare reference with NOS
//if NOS is less than reference execution is performed until doELSE.
//			or continues linear (if no doELSE)
//if greater or equal, a jump is initiated behind doELSE if present,
//			or to code address compiled by THEN
//!! after this operation former NOS is TOS now for furter handling. 
//!! DROP if not needed
// *************************************************************************
{
int32_t temp, jump;
	
    temp = (int32_t)*(gfSP-1);
	if (temp < (int32_t)*gfSP)  gfIP += 2;
	else  toElse();
    gfSP--;								//drop reference value
}


void op_DOGTCAS(void)  // ***************************************************
//( reference --)
//compiled by >CASE
//compare reference with NOS
//if NOS is less than reference execution is performed until doELSE.
//			or continues linear (if no doELSE)
//if greater or equal, a jump is initiated behind doELSE if present,
//			or to code address compiled by THEN
//!! after this operation former NOS is TOS now for furter handling. 
//!! DROP if not needed
// *************************************************************************
{
int32_t temp, jump;
	
    temp = (int32_t)*(gfSP-1);
	if (temp > (int32_t)*gfSP)  gfIP += 2;
	else  toElse();
    gfSP--;								//drop reference value
}


void toElse(void)  // *****************************************************
//common part of doIF,doCASE,doLTSCAS
// *************************************************************************
{
uint32_t jump;
	
	gfIP++;
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	gfIP = (uint8_t*)(0x20000000 | jump);//word pointer increment =4bytes!						//drop reference value
}


void op_DOELSE(void)  // ****************************************************
//( --- )
//compiled by ELSE
//if the flag of doIF != 0, doELSE manages a jump
//behind the code of the ELSE part (compiled by THEN).
//if the flag of doIF == 0, execution is continued behind doELSE.
//At the end of the ELSE part, execution is continued linar.
// *************************************************************************
{
uint32_t jump;

	gfIP++;
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	gfIP = (uint8_t*)(0x20000000 | jump);	//word pointer increment =4bytes!
}


void op_DODO(void)  // *****************************************************
//( --- )
//compiled by DO
//doDO puts the BREAK code address on the ReturnStack
// *************************************************************************
{
uint32_t jump;

    gfRP++;			//prepare Return Stack for doUNTIL, doWHILE or doAGAIN
    gfIP++;			//address next index of gfTCode
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	*gfRP = 0x20000000 | jump;
// *gfIP contains the IP jump offset relative to gfTCode base address
//this is adjusted for jump to op_RPDEC.
//CONTI needs 2 less go to doUNTIL,doWHILE,doAGAIN
//the jump address is compiled 1 less,
//because it is incremented by executeTCode() when BREAK etc. is leaving there
}

void op_DOUNTIL(void)  // **************************************************
//( flag ---)
//compiled by UNTIL
//DOUNTIL takes the flag from TOS.
//If != 0, execution goes forward behind RP-1 for linear progression
//If == 0, execution loops back to behind (doDO + 1word)
// *************************************************************************
{
    if (*gfSP == 0)  op_DOAGAIN();
	else {
        gfRP--;					//remove address for CONTI or BREAK
//            gfIP += 2;						//proceed behind RP1DEC
//bugfix 19Dec23:
            gfIP += 3;			//proceed behind doUNTIL parameter and RP1DEC
	}
    gfSP--;
}


void op_DOWHILE(void)  // **************************************************
//( flag ---)
//compiled by WHILE
//DOWHILE takes the flag from TOS.
//If == 0, execution goes forward behind RP-1 for linear progression
//If != 0, execution loops back to behind (doDO + 1word)
// *************************************************************************
{
    if (*gfSP != 0)  op_DOAGAIN();
	else {
        gfRP--;					//remove address for CONTI or BREAK
//            gfIP += 2;						//proceed behind RP1DEC
//bugfix 19Dec23:
            gfIP += 3;			//proceed behind doWHILE parameter and RP1DEC
	}
    gfSP--;
}


void op_DOAGAIN(void)  // ***************************************************
//( --- )
//compiled by AGAIN
//execution loops always back to behind (doDO + 1word)
// *************************************************************************
{
uint32_t jump;

	gfIP++;
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	gfIP = (uint8_t*)(0x20000000 | jump);	//jump behind (doDO + 1word)
											//word pointer increment =4bytes!
}


void op_DOFOR(void)  // ****************************************************
//( startIndex, endIndex increment -- )
//is RUNTIME, compiled by FOR
//3 items are copied from the DataStack to the ReturnStack:
//TOS is the increment/decrement of loop index per loop
//NOS is the loop end index for loop termination
//3rd DataStack entry -> 2nd ReturnStack entry:start index=loop index
//Furthermore the BREAK code address is put on top of the ReturnStack.
// *************************************************************************
{
uint32_t jump;
//put items to Return Stack:
    gfRP += 4;
	*(gfRP - 1) = (int32_t)*(gfSP - 2);	//actual loop counter (==start index)
	*(gfRP - 2) = (int32_t)*(gfSP -1);	//loop end index (for op_DOLOOP)
	*(gfRP - 3) = (int32_t)*gfSP;	//loop index inc/decrement(for op_DOLOOP)
	gfSP -= 3;
    gfIP++;                 //address next index of gfTCode
	jump = (uint32_t)(*gfIP << 8);
	gfIP++;
	jump |= (uint32_t)*gfIP;
	*gfRP = 0x20000000 | jump;
//this is adjusted for jump to op_RP4DEC. CONTI needs 2 less go to doLOOP
//the jump address is compiled 1 less,
//because it is incremented by executeTCode() when BREAK etc. is leaving there
}


void op_DOLOOP(void)  // ***************************************************
//( --- )
//compiled by LOOP
//counts the 2nd ReturnStack entry up or down (depends on 4th entry),
//compares the 2nd entry with the 3rd one,
//and intiates loopback to behind doFOR (if compare is in range)
//or execution goes forward behind RP-4 for linear progression.
// *************************************************************************
{
int32_t index, endIndex, deltaIndex, jump;

	index = *(gfRP - 1);
	endIndex = *(gfRP - 2);
    deltaIndex = *(gfRP - 3);
//    index += deltaIndex;					//changed 20Apr24

	if (deltaIndex >= 0)  {
    	if (index < endIndex)  {
			index += deltaIndex;			//changed 20Apr24
            *(gfRP - 1) = index; 
			gfIP++;                 //address next index of gfTCode
			jump = (uint32_t)(*gfIP << 8);
			gfIP++;
			jump |= (uint32_t)*gfIP;
			gfIP = (uint8_t*)(0x20000000 | jump);	//jump behind doFOR
    	}
    	else  {
            gfRP -= 4;						//tidy up Return Stack
            gfIP += 3;			//proceed behind doLOOP parameter and RP4DEC
    	}
	}
	else  {
		if (index > endIndex)  {
			index += deltaIndex;			//changed 20Apr24
            *(gfRP - 1) = index;
			gfIP++;							//address next index of gfTCode
			jump = (uint32_t)(*gfIP << 8);
			gfIP++;
			jump |= (uint32_t)*gfIP;
			gfIP = (uint8_t*)(0x20000000 | jump);	//jump behind doFOR
     	}
        else  {
            gfRP -= 4;						//tidy up Return Stack
//bugfix 19Dec23:
            gfIP += 3;			//proceed behind doLOOP parameter and RP4DEC
        }
    }
}


void op_RPDEC(void)  // ****************************************************
//( --- )
//compiled by WHILE and AGAIN
//At runtime it removes the top entry from the ReturnStack
// *************************************************************************
{
    gfRP--;
}


void op_RP4DEC(void)  // ***************************************************
//( --- )
//compiled by LOOP
//At runtime it removes tfour top entries from the ReturnStack
// *************************************************************************
{
    gfRP -= 4;
}

// sytem ops  ##############################################################

void op_AUTO(void)  // *****************************************************
//( --- )
//Syntax: AUTOEXE <User Thread name>
//The specified user thread is automatically executed at system start.
//To get it available at system start, the system must start from Flash.
//For details see LOAD_F and SAVE_F
// *************************************************************************
{
	gfTokenStart = gfTibPos;

	scanNext();
	if (findUserToken() == 0) {
		if (gfTokenstr[0] == 0x30)  gfAuto = 0;
		else  gfErrors |= ERR_USERSTRING;
		return;
	}
    gfAuto = gfParsed + 1;          //+1 to distinguish from gfAuto = 0
}



void op_LED(void)  // ******************************************************
// (LedBusy -- )
// if (gLedBusy == 0) LED is always OFF
// if (gLedBusy == 1) LED is always ON
// if (gLEDBusy == 2) LED blinks when RS-232 data are received 
// if (gLEDBusy == 3) LED blinks when RS-232 data are received or transmitted
//					  (default)
// *************************************************************************
{
	gLedBusy = *gfSP;
	gfSP--;
    if (gLedBusy > 3) gfErrors |= ERR_NUMVAL;			
	if (gLedBusy == 0)  GPIOB->BSRR = GPIO_BSRR_BR_3;	//PB3 User LED
	if (gLedBusy == 1)  GPIOB->BSRR = GPIO_BSRR_BS_3;
}

#ifdef USARTIO
void op_SPLIT(void)  // ****************************************************
//(type -- )
//type:'0'=USB and USART both terminal,
//'1'=USB is terminal, USART is COM I/O port
//'3'=USB is terminal, USART is MIDI interface
//the non-terminal part is used for external I/O serial communication (RX, TX)
// *************************************************************************
{
uint32_t split;
const uint8_t changeStr [] = "USART changes to ";
const uint8_t rs232Str [] = "RS232 I/O !";
const uint8_t midiStr [] = "MIDI I/O !";
const uint8_t termStr [] = "Terminal";
const uint8_t transComStr [] = "-->Transparent COM !";
const uint8_t transMidiStr [] = "-->Transparent COM<>MIDI !";

    split = *gfSP;
    gfSP --;
	if (split == gSplit)  return;

	if (split == 0)   {
		sendString(changeStr);
		sendString(termStr);
		sendTB(CRR);
	}
	else if (split == 1)   {
		sendString(changeStr);
		sendString(rs232Str);
	}
	else if (split == 3)  {
		sendString(changeStr);
		sendString(midiStr);
	}
	else if ((split == 11 ) || (split == 0x11)) {
		sendString(transComStr);
		sendTB(CRR);
		gTransparent = 1;
	}
	else if ((split == 13 ) || (split == 0x13)) {
		sendString(transMidiStr);
		sendTB(CRR);
		msDelay(2);
		gTransparent = 3;
	}

	else {
		gfErrors |= ERR_NUMVAL;
		return;
	}

	if ((split == 1) || (split == 3))  {
		sendTB(CRR);
		if (conFirm() == 0) return;
		sendTB(CRR);
		wait5us();
	}
	if ((split == 3) || (split == 13) || (split == 0x13))  {
		if (gTempBaud != 31)  {
			gTempBaud = 31;				//MIDI baudRate
			initUsart1();
		}
	}
	else  {
		if (gTempBaud != gBaud) {
			gTempBaud = gBaud;
			initUsart1();
		}
	}
	if (split <= 3)  gSplit = split;
	else  gSplit = 0;
}


void op_BAUD(void)  // *****************************************************
// (baud -- )
// parameters: ONLY FIRST 2 DIGITS of decimal baudrate are accepted:
// 11=115200,57=57600,38=38400,31=31250(MIDI),19=19200,96=9600,
// BaudRate input is always interpreted DECIMAL, independent of number base
// *************************************************************************
{
uint32_t temp;
const uint8_t newBaudStr [] = "Baud Rate changes immediately !";

	temp = *gfSP;
	gfSP--;
	if (temp == 0x96) temp = 96;
    if (temp == 0x11) temp = 11;
    if (temp == 0x19) temp = 19;
    if (temp == 0x31) temp = 31;
    if (temp == 0x38) temp = 38;
    if (temp == 0x57) temp = 57;

	if ((temp == 11) || (temp == 19) || (temp == 31) || (temp == 38) ||
											(temp == 57)|| (temp == 96)) {
		sendString(newBaudStr);
		sendTB(CRR);
		if (conFirm())  {
			gBaud = temp;
			gTempBaud = temp;
			initUsart1();
		}
    }
    else  {
		gfErrors |= ERR_NUMVAL;
	}
}
#endif

void op_QUEST(void)  // *****************************************************
// ( -- )
//  displays a list of active system coonfiguration
// **************************************************************************
{
uint32_t k, r, q;

const uint8_t typeStr [] = "CForth for NucleoF042 v";
const uint8_t copyStr [] = " (c)2020-25 W.Schemmert";
const uint8_t cpuSpeedStr [] = "  CPUclock=48MHz-HS";
const uint8_t stackStr [] = "Stack=";
const uint8_t threadStr [] = "CodeThread=";
const uint8_t emptyStr [] = "Empty_";
const uint8_t flashStr [] = "Flash";
const uint8_t projectStr [] = "Project";
const uint8_t autoStr [] = "  AUTOEXE=";
const uint8_t baudStr [] = "  BaudR=";
const uint8_t splitStr [] = "  Split=";
const uint8_t ledStr [] = "  LEDmode=";
const uint8_t debugStr [] = "  BackOP_O";
const uint8_t freqStr [] = "   FREQ=";
const uint8_t pwm1Str [] = "Hz PWM1=";
const uint8_t pwm2Str [] = " PWM2=";
const uint8_t offStr [] = "OFF";
const uint8_t motStr [] = "   MOT:";
const uint8_t tstepStr [] = "Step=";
const uint8_t nstepsStr [] = "ms NSTP=";
const uint8_t dcmotStr [] = "DC/PWM1";

    sendString(typeStr);
    sendTB(HARDWARE_VERSION + 0x30);
    sendTB('.');
    sendNS32(REVNUM);
    sendString(copyStr);
	sendString(cpuSpeedStr);
#ifdef EXTCLOCK_8M
	sendTB('E');
#else
	sendTB('I');
#endif
	sendTB('-');
    sendTB(CRR);

    sendTB('D');
    sendTB('a');
    sendTB('t');
    sendTB('a');
    sendString(stackStr);
    sendX32((uint32_t)(gfDataStack + 3));
    sendTB('R');
    sendTB('e');
    sendTB('t');
    sendTB('u');
    sendTB('r');
    sendTB('n');
    sendString(stackStr);
    sendX32((uint32_t)gfReturnStack);
    sendString(threadStr);
    sendX32((uint32_t)gfTCode);
    sendTB('U');
    sendTB('s');
    sendTB('e');
    sendTB('r');
    sendTB('=');
    sendX32((uint32_t)gfUserHeader);
    sendTB(CRR);

	sendProjNum(gProject);
	sendString(autoStr);
	if (gfAuto == 0 )  {
		sendTB('n');
		sendTB('o');
		sendTB('n');
		sendTB('e');
	}
	else  {	
		for (k = 0;k < 8;k++)  {
			if (gfUserHeader[gfAuto-1].name[k] == 0)  break;
			sendTB(gfUserHeader[gfAuto-1].name[k]);
		}
	}
#ifdef USARTIO
	sendString(baudStr);
	sendNS32(gTempBaud);
	if (gTempBaud == 31)  {
		sendTB('2');
		sendTB('5');
		sendTB('0');
	}
	else  {
		if (gTempBaud == 11)  {
			sendTB('5');
			sendTB('2');
		}
		if (gTempBaud == 19)  sendTB('2');
		if (gTempBaud == 38)  sendTB('4');
		if (gTempBaud == 57)  sendTB('6');
		sendTB('0');
		sendTB('0');
	}

	sendString(splitStr);
	sendTB(gSplit | 0x30);
#endif
	sendString(ledStr);
	sendTB(gLedBusy | 0x30);
	sendString(debugStr);
	if (gfBackProcess == 0)  {
		sendTB('F');
		sendTB('F');
	}
	else  sendTB('N');
	sendTB(CRR);

	sendTB('S');
	sendTB('P');
	sendTB('I');
	sendTB('=');
	sendNS32(gfSpiMode);
	sendTB(',');
	if (gfTempNumBase==16)  {
		send0X32(gfSpiBits);
		sendTB(',');
		send0X32(gfSpiClock);
	}
	else  {
		sendNS32(gfSpiBits);
		sendTB(',');
		sendNS32(gfSpiClock);
	}
	sendString(freqStr);
//reference for FREQ and PWM is 24KHz
	if (gfFreq <= 0) sendNS32(0);
	else  {
		k = gfFreq * 2000;
		q = 48000000 / k;
		sendNS32(q);
		r = (48000000 - (q * k)) * 10;
		sendTB('.');
        q = r / k;
		sendNS32(q);
		r = (r - (q * k)) * 10;
        q = r / k;
		r = (r - (q * k)) * 10;
        k = r / k;
        if (k >= 5) q++;
		sendNS32(q);
	}

	sendString(pwm1Str);
	if (gfPWM1 < 0) sendString(offStr);
	else {
		if (gfTempNumBase==16)  send0X32(gfPWM1);
		else  sendNS32(gfPWM1);
	}
	sendString(pwm2Str);
	if (gfPWM2 < 0) sendString(offStr);
	else {
		if (gfTempNumBase==16)  send0X32(gfPWM2);
		else  sendNS32(gfPWM2);
	}
#ifndef USARTIO
	sendString(motStr);
	if (gfUserHeader[4].toCompile == -1)  {
		if (GPIOA->ODR & 1) sendTB('-');			//PA0 High = CCW
		sendString(dcmotStr);
	}
	else  {
		sendString(tstepStr);
		sendNS32(gfStepSpeed);
		sendString(nstepsStr);
		sendNS32(gfUserHeader[4].toCompile);
	}
#endif
	sendTB(CRR);
}

void op_SAVEFLASH(void)  // ************************************************
// save this project state in microcontroller internal flash
// *************************************************************************
{
uint32_t n;

	n = checkProjectNum(1);
	if (n > 2) return;

	gProject = n;
	if (n == 1) pushFlashProject(1);
//complicated procedure because no temp buffer: don't destroy Proj#1
	else if (n == 2) {
		pushFlashProject(2);	//save content
		popFlashProject(1);		
		gProject = 2;			//enter gProject=2 into flash PAGE26
		pushFlashProject(1);	//excluasively change gProject for power-on
		popFlashProject(2);		//reload content
	}
	else  {
		popFlashProject(1);
		gProject = 0;			//enter gProject=0 into flash PAGE26
		pushFlashProject(1);	//excluasively change gProject for power-on
	}
	if (n ==0 ) emptyForth();	//refresh "gfNextNameCompi" and gfAuto
	else op_ABORT();			//for op_USER etc.
    gProject = n;				//use it
}


void op_LOADFLASH(void)  // ************************************************
// reload previously saved project from microcontroller internal flash
// *************************************************************************
{
uint32_t n;

	n = checkProjectNum(0);
	if (n > 2) return;

    if (n == 0)  emptyForth();
	else 	popFlashProject(n);
	gProject = n;				//override gProject read from flash PAGE26
	postLoad();
}


uint32_t conFirm(void)  // *************************************************
//save some code length
// *************************************************************************
{
uint32_t k;
const uint8_t confirmStr [] = "confirm w.upper case 'Y'";

	sendString(confirmStr);
	while ((k= pullTermRxBuf()) == 0);
	if ((k & 0x000000FF) != 'Y')  {
		sendString(gfCancelStr);
		return 0;
	}
    sendTB(CRR);
	return 1;
}


void sendProjNum(uint32_t  projNum)  // ************************************
// reload previously saved project from microcontroller internal flash
// *************************************************************************
{
const uint8_t emptyStr [] = "Empty_";
const uint8_t projectStr [] = "Project";
	
	if (projNum == 0)  sendString(emptyStr);
	sendString(projectStr);
	if (projNum)  {
		sendTB('#');
		sendTB(projNum + 0x30);
	}
}


uint32_t checkProjectNum(uint32_t mode)  // *********************************
// *************************************************************************
{
uint32_t n;
const uint8_t loadStr [] = "Load ";
const uint8_t saveStr [] = "Save ";

    n = *gfSP;
	gfSP--;

	if (n > 2)  {
		gfErrors |= ERR_NUMVAL;
		return 0xFFFFFFFF;
	}

	if (mode == 0) sendString(loadStr);
	else  sendString(saveStr);
	sendProjNum(n);
	sendTB(SPACE);
	sendTB('?');
	sendTB(SPACE);
	if (conFirm() == 0)  return 3;
	return n;
}


void checkBackNum(void)  // ***********;;;;*********************************
//save some code length
// *************************************************************************
{
	if (gfBackNumValid)  {
		gfBackNumValid = 0;
		gfSP++;
		*gfSP = gfBackProcData;
	}
}

// ### END #################################################################
