// CForth LowPower project ***************************************************
//
// Author and (c)2020-25 Wolfgang Schemmert. The copyright claim is restricted
// to this file and the depending files "main.h","cforth.c","cforth.h",
// and "globaldefines.h"
// Other depending material is published, copyrighted and licensed
// by STMicroelectronics and Segger GmbH
//
// Contact: <www.midi-and-more.de>
// A detailled hardware construction and operation manual
// is provided at this website
//
// ****************************************************************************
//      This program is free software: you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation, either version 3 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//      Licence: GPL-3
//
// ****************************************************************************

#include "stm32f0xx.h"
#include "globaldefines.h"
#include "main.h"

int main(void)    // ********************************************************
// **************************************************************************
{
uint32_t recv;
START:

    RCC->AHBENR |= RCC_AHBENR_GPIOAEN;  // Enable Port A clock
    RCC->AHBENR |= RCC_AHBENR_GPIOBEN;  // Enable Port B clock

//	initGPIO();
    GPIOA->MODER = 0x00000000;      //primarily all input
    GPIOB->MODER = 0x00000000;      //primarily all input
#ifdef USARTIO
    GPIOA->PUPDR = 0x0182AA8A;      //primarily all inputs pulldown 
									//except PA15,14,13,10,9,2-PA12=PullUP
#else
    GPIOA->PUPDR = 0x019AAA8A;      //primarily all inputs pulldown 
									//except PA15,14,13,2-PA12=PullUP
#endif
    GPIOB->PUPDR = 0xAAAAAAAA;      //primarily all inputs pulldown

//pin config repeated specifically:
	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR12;		//PA12->input pull-up
	GPIOA->PUPDR |= GPIO_PUPDR_PUPDR12_0;		//USART: mode check
												//MOTOR:kill AUTOEXE
 	GPIOB->PUPDR &= ~(GPIO_PUPDR_PUPDR3);	//pull-down off, wasts current
    GPIOB->MODER &= ~(GPIO_MODER_MODER3);
    GPIOB->MODER |= GPIO_MODER_MODER3_0 ;	//set PortB3 (User LED) as output
    GPIOB->BSRR =  GPIO_BSRR_BS_3;			//PB3 (User LED) ON

/*  
//useful for debugging
	//MCO is AF0 on PortA
    GPIOA->MODER &= ~(GPIO_MODER_MODER8_0);
    GPIOA->MODER |= GPIO_MODER_MODER8_1;
    GPIOA->AFR[1] &= 0xFFFFFFF0;        //PA8 is MCO
    GPIOA->AFR[1] |= 0x00000000;
*/

	SetSysClock();

    gLEDCOUNT = 0;
    gDELAYCOUNT = 0;
    gTIMEOUT = 0;
	gfNumBase = DEFAULT_NUMBASE;
	gfTempNumBase = DEFAULT_NUMBASE;
/*
#ifdef USARTIO
	gBaud = DEFAULT_BAUD;
	gTempBaud = DEFAULT_BAUD;
#endif
*/
	gProject = 0;				//temporarily
	gfAuto = 0;
	gLedBusy = 2;
	gfRandom = 314159265;		//seed for xorshift32 algorithm

    SysTick_Config(SystemCoreClock / 1000);	
								//1 ms for any SystemCoreClock
								//SystemCoreClock is derived from SYSCLOCK=HSE

    msDelay(50);

//init I/O state  *************************
    op_ABORT();					//reset default Forth system data
	popFlashProject(0);			//loads only gBaud, gfNumBase and gProject

    msDelay(2);
 //init communication  ********************
	initUsart2();
    msDelay(2);

#ifdef USARTIO
	gSplit = 0;
	gBaud = DEFAULT_BAUD;						//= 115.2 kBaud
	gTempBaud = DEFAULT_BAUD;
	gTransparent = ~(GPIOA->IDR >> 12) & 0x00000001; 
									//check PA12, if LOW: gTransparent = 1
	gTransEnd = 0;
//result:	gTransparent =0: CForth
//			gTransparent =1: USB_COM <-> USART_RS232 interface

	initUsart1();
#endif

	popFlashProject(gProject);
	postLoad();

#ifdef USARTIO
    while(1) {
		if (gTransparent) {		//if gTransparent ==1:
								//COM transparent (PA12=Low,or 11 SPLIT)
								//if gTransparent ==3:
								//COM transparent (13 SPLIT)
			recv = pullUsbRxBuf();
			if (recv)  sendB((uint8_t)(recv & 0x000000FF));
			recv = pullUsartRxBuf();
			if (recv)  sendUsbByte((uint8_t)(recv & 0x000000FF));
		}
		else  {								//gTransparent==0: CForth mode
			quit();
			query();
			if (gfTibMax == 0) continue;
			eval();
		}
    }
#else
    while(1) {
		quit();
		if (query() == 0) break;
		if (gfTibMax == 0) continue;
		eval();
    }
#endif
}


// ##########################################################################

// Misc Low Level Routines
//************************

#ifdef EXTCLOCK_8M
//always PLLCLOCK
void SetSysClock(void)  // ******************************************
// *************************************************************************
{
__IO uint32_t StartUpCounter = 0, HSEStatus = 0;

//SYSCLK, HCLK, PCLK configuration ----------------------------------------
//Enable HSE 
  RCC->CR |= ((uint32_t)RCC_CR_HSEON);

//Wait until HSE is ready -- if Time out is reached exit 
  do  {
    HSEStatus = RCC->CR & RCC_CR_HSERDY;
    StartUpCounter++;
  } while((HSEStatus == 0) && (StartUpCounter != HSE_STARTUP_TIMEOUT));

  if ((RCC->CR & RCC_CR_HSERDY) != RESET)  { //RESET=0 defined in stm32f0xx.h
//Enable Prefetch Buffer and set Flash Latency 
//for CLOCK_48M
    FLASH->ACR = FLASH_ACR_PRFTBE | FLASH_ACR_LATENCY;

	RCC->CFGR &= ~(RCC_CFGR_HPRE);		//clear all HPRE options
	RCC->CFGR |=  RCC_CFGR_HPRE_DIV1;

//PCLK = HCLK 
    RCC->CFGR |= (uint32_t)RCC_CFGR_PPRE_DIV1;

// useful for debugging
//	RCC->CFGR |=  RCC_CFGR_MCO_SYSCLK;	
//	RCC->CFGR |= RCC_CFGR_MCO_PRE_16;	//set for project to 48M/16=3MHz

	RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_PLLSRC | 
									RCC_CFGR_PLLXTPRE | RCC_CFGR_PLLMULL));

//PLL configuration = HSE with 8.0 MHz: HSE * 6 * 48MHz
    RCC->CFGR |= (uint32_t)(RCC_CFGR_PLLSRC_PREDIV1 | 
							RCC_CFGR_PLLXTPRE_PREDIV1 | RCC_CFGR_PLLMULL6);

//Enable PLL 
    RCC->CR |= RCC_CR_PLLON;

//Wait till PLL is ready
    while((RCC->CR & RCC_CR_PLLRDY) == 0) ;

//Select PLL as system clock source
    RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_SW));
    RCC->CFGR |= (uint32_t)RCC_CFGR_SW_PLL;

//Wait till PLL is used as system clock source
    while ((RCC->CFGR & (uint32_t)RCC_CFGR_SWS) != (uint32_t)RCC_CFGR_SWS_PLL) ;

	RCC->CFGR3 = RCC_CFGR3_USART2SW_0;	//SYSCLK selected for USART2
  }
}

#else

void SetSysClock(void)  // ******************************************
// *************************************************************************
{
__IO uint32_t StartUpCounter = 0, HSIStatus = 0;

//SYSCLK, HCLK, PCLK configuration ----------------------------------------
//Enable HSI 
  RCC->CR &= ~((uint32_t)RCC_CR_HSEON);			//precaution:switch OFF
  RCC->CR |= ((uint32_t)RCC_CR_HSION);			//should be active by default

//Wait until HSI is ready -- if Time out is reached exit 
  do  {
    HSIStatus = RCC->CR & RCC_CR_HSIRDY;
    StartUpCounter++;
  } while((HSIStatus == 0) && (StartUpCounter != HSE_STARTUP_TIMEOUT));

//  if ((RCC->CR & RCC_CR_HSIRDY) != RESET)  { //RESET=0 defined in stm32f0xx.h
//Enable Prefetch Buffer and set Flash Latency 
//for CLOCK_48M
    FLASH->ACR = FLASH_ACR_PRFTBE | FLASH_ACR_LATENCY;

	RCC->CFGR &= ~(RCC_CFGR_HPRE);				//clear all HPRE options
	RCC->CFGR |=  RCC_CFGR_HPRE_DIV1;

//PCLK = HCLK 
    RCC->CFGR |= (uint32_t)RCC_CFGR_PPRE_DIV1;

//useful for debugging
//	RCC->CFGR |=  RCC_CFGR_MCO_SYSCLK;	
//	RCC->CFGR |= RCC_CFGR_MCO_PRE_16;	//set for project to 48M/16=3MHz

	RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_PLLSRC | RCC_CFGR_PLLXTPRE 
								| RCC_CFGR_PLLMULL)); //select PLL input:HSI/2
//  CLOCK_48M
//PLL configuration = HSI/2 = 4.0 MHz: HSI * 12 * 48MHz
    RCC->CFGR |= (uint32_t)( RCC_CFGR_PLLMULL12);

//Enable PLL 
    RCC->CR |= RCC_CR_PLLON;

//Wait till PLL is ready
    while((RCC->CR & RCC_CR_PLLRDY) == 0) ;

//Select PLL as system clock source
    RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_SW));
    RCC->CFGR |= (uint32_t)RCC_CFGR_SW_PLL;

//Wait till PLL is used as system clock source
    while ((RCC->CFGR & (uint32_t)RCC_CFGR_SWS) != (uint32_t)RCC_CFGR_SWS_PLL) ;

	RCC->CFGR3 = RCC_CFGR3_USART2SW_0;	//SYSCLK selected for USART2
}

#endif
// ************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE* //

void LED_Off(void)  // ******************************************************
// built as subroutine to supply service to "usb-fs-042.c" hardware independent
// **************************************************************************
{
		if (gLedBusy  >= 2) {
			GPIOB->BSRR = GPIO_BSRR_BR_3;	// clear bit PB3,LED OFF
			gLEDCOUNT = 4; //ca 300 milliseconds
		}
}


void msDelay(uint32_t count)   // *******************************************
//blocking delay by 'count' milliseconds
// **************************************************************************
{
    gDELAYCOUNT = count;
    while(gDELAYCOUNT)  servSystem();		//unconditionally in this case
}


void wait5us(void)   // *****************************************************
// blocking delay ca 5us
// **************************************************************************
{
uint32_t countdown;

	countdown = 51;
    while (countdown)   {
        countdown-- ;
    }
}


void servSystem(void)   // ***************************************************
//not very time critical but permanently repeated functions for system control
//exclusively called in pullTermRxBuf
//this way it is regularily served when idle waiting for external input
//but watchdog resets when it hangs somewhere else
//slowed down by SysTick to 50 times/second
// ***************************************************************************
{
	if (gSYSCOUNT)  return;		//countdown in SysTick_Handler()

//else if if (gSYSCOUNT==0 )50 Hz service interval:  *************************
    gSYSCOUNT = 50;         //restart 20Hz service interval

//LED control   28Dec24 LED blink behaviour changed for better user access
	if  (gLEDCOUNT)  {
		gLEDCOUNT--;
		if (gLEDCOUNT==0)  {
			if (gLedBusy >= 2)  	GPIOB->BSRR = GPIO_BSRR_BS_3;
											// Set bit PB3 (User LED) ON again
		}
	}
}


void popFlashProject(uint32_t mode)   //*************************************
// **************************************************************************
{
uint32_t k, n;
uint32_t  *flashPtr;
uint32_t *sramPtr;

//	emptyForth();		//load default empty project

//emptyForth() already updated in popFlashPage26() (default system parameters)
	if (mode == 0) flashPtr = (uint32_t *)PAGE26;
								//only returns gProject,gBaud and gfNumBase
//	else if (mode == 2) flashPtr = (uint32_t *)PAGE29;
								//only returns gProject,gBaud and gfNumBase
	else  {
		if (mode == 1) flashPtr = (uint32_t *)PAGE26;
		else flashPtr = (uint32_t *)PAGE29;
	}
//4 words=16bytes for system and peripherial parameters
	k = ~(*flashPtr);			//DUMMY cannot be==0 if project saved in Flash
	flashPtr++;
	gProject = (k & 0x00000003) -1;	//finally only relevant for PAGE26
	n =  ~(*flashPtr);			//cannot be==0 if project saved in Flash
	flashPtr++;
	if ((n | k) == 0)  {		//project Flash is virgin
		emptyForth();		//load default empty project if virgin Flash
//		if (mode == 0) return;
		return;
	}

//Flash is not virgin,load basic parameters,interpreted as "globals" in PAGE26
		if (n & 0x00000001)  gfNumBase = 16;	
		else gfNumBase = 10;					//virgin Flash defauolt
		gfTempNumBase = gfNumBase;
#ifdef USARTIO
		gBaud = (n >> 1) & 0x000007F;			//bits 1...7
		if (gBaud == 0)  gBaud = DEFAULT_BAUD;	//virgin Flash default
		gTempBaud = gBaud;
#endif
//Bit8 still available	

	if (mode)  {
//from here only pre-saved project#1 or #2 are handled
		gfSpiMode = (n >> 9) & 0x00000003;	
		gfSpiBits = ((n >> 11) & 0x0000001F) + 4;	//range 4...32
		gfSpiClock = (n >> 16) & 0x0000FFFF;

		k = ~(*flashPtr);
		flashPtr++;
		gfAuto = (k & 0x0000FFFF);
		gfFreq = (k >> 16) & 0x0000FFFF;

		k = ~(*flashPtr);
		flashPtr++;
		n = k & 0x0000FFFF;
		if (n > 2048) n |= 0xFFFF0000;			//manual cast to int32_t
		gfPWM1 = (int32_t)n;
		if (gfFreq) {
			subFREQ();
			subPWM1();
			n = (k >> 16) & 0x0000FFFF;
			if (n > 2048) n |= 0xFFFF0000;		//manual cast to int32_t
			gfPWM2 = (int32_t)n;
			subPWM2();
		}

//loading is possible over page limits.
//gfUserHeader[] and gfTcode[] occupy a coherent data space in flash
		sramPtr = (uint32_t *) &(gfUserHeader[0]);
		for (k = 0;k < (MAXUSER * 4);k++)  {	
									//380 words=1520 bytes=ca.1.5pages
			*sramPtr = ~(*flashPtr);
			sramPtr++;
			flashPtr++;
		}
		sramPtr = (uint32_t *) &(gfTCode[0]);
		for (k = 0;k < 0x180;k++)  {	//1536(0x600) bytes = 1.5 pages
			*sramPtr = ~(*flashPtr);
			sramPtr++;
			flashPtr++;
		}
	}

//emergency brake to avoid AUTO deadlock
#ifndef USARTIO
	if ((GPIOA->IDR & 0x1000) == 0) gfAuto = 0;	//if PA12 (jumper pin) is low
	GPIOA->PUPDR &= ~GPIO_PUPDR_PUPDR10;		//HiZ
	GPIOA->PUPDR |= GPIO_PUPDR_PUPDR10_1;		//pull-down = default state
#else
	if ((GPIOA->IDR & 0x0400) == 0) gfAuto = 0;	//if PA10 (RX input)is low
#endif
}

void postLoad(void)   //*****************************************************
// **************************************************************************
{
	op_ABORT();
	if (gProject == 0)  gfAuto = 0;
#ifdef USARTIO
	if (gTransparent == 0)  {
#endif
		op_QUEST();
		if (gProject)  {
			if (gfAuto) {
				gfParsed = gfAuto-1;	//-1 to distinguish from empty AUTOEXE=0
				interpretUserToken();
			}
		}
#ifdef USARTIO
	}
#endif
}


void pushFlashProject(uint32_t project)   //*********************************
// **************************************************************************
{
uint32_t k;
uint16_t  *flashPtr;
uint32_t *sramPtr;

//unlock flash
#define KEY1    0x45670123      //mandatory unlock key defined by STM
#define KEY2    0xCDEF89AB
    while ((FLASH->SR & FLASH_SR_BSY) !=0);
								//wait until no operation is going on
    if ((FLASH->CR & FLASH_CR_LOCK) != 0) {
            FLASH->KEYR = KEY1;
            FLASH->KEYR = KEY2;
    }

//erase Page26 (project==1) or Page29 (project==2)
    FLASH->CR |= FLASH_CR_PER;
    if (project == 1) FLASH->AR = PAGE26;
	else FLASH->AR = PAGE29;
    FLASH->CR |= FLASH_CR_STRT;
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) !=0)  FLASH->SR |= FLASH_SR_EOP;
    FLASH->CR &= ~FLASH_CR_PER;

    FLASH->CR |= FLASH_CR_PG;
//write system parameters and gfUserHeader[] into page29
//programming Flash is possible only by 16bit half-words
//first programmed 16bit half-word will be read as lower 16bits
//                                         when data are read as 32bit words!!
//second programmed 16bit half-word will be read as higher 16bits
//                                         when data are read as 32bit words!!
	if (project == 1) flashPtr = (uint16_t *)PAGE26;
	else flashPtr = (uint16_t *)PAGE29;

    //4 words=8 half-words=16bytes for system and peripheral parameters
    *flashPtr = (uint16_t)(~(gProject+1) & 0x0000FFFF);//read = lower 16bit
							//cannot be zero, used for Flash virgin test
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
    flashPtr++;
    *flashPtr = 0;//DUMMY read = upper 16bit, used for Flash virgin test
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
    flashPtr++;

	if (gfNumBase == 10) k = 0;
    else k = 1;
	k |= ((gBaud & 0x0000007F) << 1); // | ((gRsTerm & 1) << 8);
	k |= ((gfSpiMode & 0x0000003)<< 9) ; 
    k |= (((gfSpiBits - 4) & 0x0000001F) << 11); //burn range = 0...28
	*flashPtr = (uint16_t)(~k & 0x0000FFFF); //program first=read lower16bit
	while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
	if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
	flashPtr++;
	k =  gfSpiClock;
	*flashPtr = (uint16_t)(~k & 0x0000FFFF); //program last=read upper16bit
	while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
	if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
	flashPtr ++;	

    *flashPtr = (uint16_t)(~gfAuto & 0x0000FFFF);	//read as lower 16bits
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
    flashPtr++;
   *flashPtr = (uint16_t)(~gfFreq & 0x0000FFFF);	//read as upper 16bits
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
    flashPtr++;

	k = (uint32_t) gfPWM1 ;
	*flashPtr = (uint16_t)(~k & 0x0000FFFF); //program first=read lower16bit
	while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
	if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
	flashPtr ++;
	k = (uint32_t) gfPWM2 ;
	*flashPtr = (uint16_t)(~k & 0x0000FFFF); //program last=read upper16bit
	while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
	if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
	flashPtr ++;

//gfUserHeader[] and gfTcode[] occupy a coherent data space in flash
    sramPtr = (uint32_t *) &(gfUserHeader[0]);//here:252words=1008bytes+16=1024
    for (k = 0;k < 252;k++)  { //each entry 16bytes ->first 63 entries
        *flashPtr =(uint16_t)(~(*sramPtr) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        *flashPtr =(uint16_t)((~(*sramPtr) >> 16) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        sramPtr++;
    }
    FLASH->CR &= ~FLASH_CR_PG;

//erase Page27 (project==1) or Page30 (project==2)
    FLASH->CR |= FLASH_CR_PER;
	if (project == 1) FLASH->AR = PAGE27;
	else FLASH->AR = PAGE30;
    FLASH->CR |= FLASH_CR_STRT;
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) !=0)  FLASH->SR |= FLASH_SR_EOP;
    FLASH->CR &= ~FLASH_CR_PER;

    FLASH->CR |= FLASH_CR_PG;
	if (project == 1) flashPtr = (uint16_t *)PAGE27;
	else flashPtr = (uint16_t *)PAGE30;
//write remaining part of gfUserHeader:MAXUSER=(95-63) *4=128 words total
    for (k = 0;k < 128;k++)  { //each entry 16bytes --> last 32entries
        *flashPtr =(uint16_t)(~(*sramPtr) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        *flashPtr =(uint16_t)((~(*sramPtr) >> 16) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        sramPtr++;
    }
//flashed up to 0x8007A00
//write first part of gfTCode into page 27(project==1) or page30(project==2)
    sramPtr = (uint32_t *) &(gfTCode[0]);	//WORD addressed!
    for (k = 0;k < 0x80;k++)  {				//128words=512bytes=0.5 page
        *flashPtr =(uint16_t)(~(*sramPtr) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        *flashPtr =(uint16_t)(~(*sramPtr >> 16) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        sramPtr++;
    }
    FLASH->CR &= ~FLASH_CR_PG;

//erase Page28 (project==1) or Page31 (project==2)
    FLASH->CR |= FLASH_CR_PER;
	if (project == 1) FLASH->AR = PAGE28;
	else FLASH->AR = PAGE31;
    FLASH->CR |= FLASH_CR_STRT;
    while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
    if ((FLASH->SR & FLASH_SR_EOP) !=0)  FLASH->SR |= FLASH_SR_EOP;
    FLASH->CR &= ~FLASH_CR_PER;

    FLASH->CR |= FLASH_CR_PG;
//write second half of gfTCode into  28(project==1) or page31(project==2)
	if (project == 1) flashPtr = (uint16_t *)PAGE28;
	else flashPtr = (uint16_t *)PAGE31;
    sramPtr = (uint32_t *)gfTCode + 0x80;	//start addr = 0x200bytes
    for (k = 0;k < 0x100;k++)  { //0x400bytes= 1 page, word read,halfWord burn
        *flashPtr =(uint16_t)(~(*sramPtr) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        *flashPtr =(uint16_t)(~(*sramPtr >> 16) & 0x0000FFFF);
        while ((FLASH->SR & FLASH_SR_BSY) != 0) ;
        if ((FLASH->SR & FLASH_SR_EOP) != 0)  FLASH->SR |= FLASH_SR_EOP;
        flashPtr++;
        sramPtr++;
    }
//flash is 100% filled
//terminate
    FLASH->CR &= ~FLASH_CR_PG;
//lock flash
    FLASH->CR |= FLASH_CR_LOCK;
    msDelay(10);
}

//###########################################################################

//USART Support
//*************
void initUsart2(void)  // ****************************************************
// if USB is used with TSSOP package, USART2 is the only available USART
// USART2-TxD is PortA,2
// USART2-RxD is PortA,15
// **************************************************************************
{
uint16_t baudset;

    RCC->AHBENR |= RCC_AHBENR_GPIOAEN ;     // PortA clock enable
    RCC->APB1ENR |=  RCC_APB1ENR_USART2EN ; // USART2 clock enable

    GPIOA->MODER &= ~(GPIO_MODER_MODER2_0 | //Set pin 2,3 to alternate function
                      GPIO_MODER_MODER15_0 ) ;  // 10=Alternate function mode
    GPIOA->MODER |= (GPIO_MODER_MODER2_1 |  //Set pin 2,3 to alternate function
                     GPIO_MODER_MODER15_1 ) ;  // 10=Alternate function mode
    //GPIOA->OTYPER is push-pull by default
    //GPIOA->OSPEEDR is 2 MHz by default
    //GPIOA->PUPDR is HiZ by default (except PA13,14,15)

//USART2 is AF1
    GPIOA->AFR[0] = 0x00000100;
    GPIOA->AFR[1] = 0x10000000;

    gSTLINK_RXHEAD = gSTLINK_RXBUF;
    gSTLINK_RXTAIL = gSTLINK_RXBUF;

	USART2->BRR =  0x01A0;		//115200Baud: 26.0 at 48 MHz APB1 Clock;
    USART2->CR2 = 0;				//default:1 Stopbit
    USART2->CR3 = 0;				//default:no Handshake
//the TXIE flag will be set when the first byte is going to be transmitted
    USART2->CR1 = USART_CR1_UE | USART_CR1_RXNEIE |
                  USART_CR1_TE | USART_CR1_RE;  //8Bit, no Parity,RX&TX Int ON
    NVIC_EnableIRQ(USART2_IRQn);
}


void sendUsbByte(uint8_t data) // *************************************************
//send 1 raw byte
// **************************************************************************
{
    while(!(USART2->ISR & USART_ISR_TXE));
    USART2->TDR = data;
	if (gLedBusy == 3) LED_Off();
}

#ifdef USARTIO
void initUsart1(void) // ****************************************************
// USART1-TxD is PortA,9
// USART1-RxD is PortA,10
// **************************************************************************
{
uint16_t baudset;

    RCC->APB2ENR |=  RCC_APB2ENR_USART1EN ;   // USART1 clock enable
//set PA9/PA10 in alternate function mode
    GPIOA->MODER &= ~(GPIO_MODER_MODER9_0 | GPIO_MODER_MODER10_0);
    GPIOA->MODER |= (GPIO_MODER_MODER9_1 | GPIO_MODER_MODER10_1);

//USART1 is AF1 on PortA
    GPIOA->AFR[1] &= 0xFFFFF00F;        //PA9 is TX, PA10 is RX
    GPIOA->AFR[1] |= 0x00000110;

	gUSART_RXHEAD = gUSART_RXBUF;
	gUSART_RXTAIL = gUSART_RXBUF;
	gMIDI_HEAD = 0;
	gMIDI_TAIL = 0;

    switch(gTempBaud) {
    case 96:
        baudset = 0x1380;       //9600Baud: 312.0 at 48 MHz APB2 Clock
        break;
    case 19:
        baudset = 0x09C0;       //19200Baud: 156.0 at 48 MHz APB2 Clock
        break;
    case 31:
        baudset = 0x0600 ;      //MIDI=31250Baud: 96.0 at 48 MHz APB2 Clock
        break;
    case 38:
        baudset = 0x04E0 ;      //38400Baud: 78.0 at 48 MHz APB2 Clock
        break;
    case 57:
        baudset = 0x0340;       //57600Baud: 52.0 at 48 MHz APB2 Clock
        break;
    case 11:
    default:
        baudset = 0x01A0;       //default (virgin flash)
                  				//115200Baud: 26.0 at 48 MHz APB2 Clock
    }
    USART1->BRR = baudset;		//default 115200Baud:26.0 at 48 MHz APB2 Clock

	USART1->CR2 = USART_CR2_TXINV;//1 Stopbit, TX level inverted
    USART1->CR3 = 0;			//default:no Handshake
    USART1->CR1 = USART_CR1_UE|USART_CR1_RXNEIE|USART_CR1_TE|USART_CR1_RE;
								//8Bit, no Parity, RX Int
    NVIC_EnableIRQ(USART1_IRQn);
}


void sendB(uint8_t data) // *******************************************
//send 1 raw byte
// **************************************************************************
{
    while(!(USART1->ISR&USART_ISR_TXE));
    USART1->TDR  = data;
	if (gLedBusy == 3)  LED_Off();
}


void sendTB(uint8_t data) // ************************************************
//send one byte via serial terminal port (USB and/or USART)
//0=USB and USART both terminal, 1+3= USB is terminal, 2+4=USART is terminal
// **************************************************************************
{
    if ((gSplit == 0) && (gTransparent == 0)) {
        sendB(data);
		sendUsbByte(data);
    }
    else sendUsbByte(data);
                            //USB is terminal(virtual COM),USART is serial I/O
}


void sendIOB(uint8_t data) // ************************************************
//send one byte via serial I/O
// **************************************************************************
{
    if (gSplit) {						//USB is terminal(virtual COM),
        sendB(data); 					//USART is serial I/O
    }
	else  gfErrors |= ERR_CONTEXT;		//gSplit=0
}

#else

void sendTB(uint8_t data) // *************************************************
//send 1 raw byte
// **************************************************************************
{
    while(!(USART2->ISR&USART_ISR_TXE));
    USART2->TDR = data;
	if (gLedBusy == 3)  LED_Off();
}
#endif

void sendNibble(uint8_t data) // ********************************************
// **************************************************************************
{
uint8_t asciiNibble;

    asciiNibble = data+0x30;
    if (asciiNibble>0x39) asciiNibble+= 7;
	sendTB(asciiNibble);
}


void sendString(const uint8_t * str) // *************************************
// **************************************************************************
{
 uint8_t byte;

    while  ((byte = *str++) != 0)  sendTB(byte);
}


void sendX8(uint32_t data) // *********************************************
// send bits 0...7 in hex format (without leading 0x)
// **************************************************************************
{
    uint8_t temp;

    temp= (data & 0xF0) >> 4;
    temp = temp  +0x30;
    if (temp > 0x39) temp+= 7;
    sendTB(temp);
    temp= data & 0x0F;
    temp = temp + 0x30;
    if (temp > 0x39) temp += 7;
    sendTB(temp);
    sendTB(0x20);
}


void sendX32(uint32_t data) // **********************************************
// send 1 word (32bit) in hex format (without leading 0x)
// Leading zeroes are suppressed
// **************************************************************************
{
uint8_t nibble;
    nibble = (uint8_t) ((data>>28) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) ((data>>24) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) ((data>>20) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) ((data>>16) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) ((data>>12) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) ((data>>8) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) ((data>>4) & 0x0000000F);
    sendNibble(nibble);
    nibble = (uint8_t) (data & 0x0000000F);
    sendNibble(nibble);
    sendTB(0x20);

}


void sendNX32(uint32_t data) // **********************************************
// send 1 word (32bit) in hex format (without leading 0x)
// **************************************************************************
{
uint8_t nibble, noLead;
    noLead = 0;
    nibble = (uint8_t) ((data>>28) & 0x0000000F);
    if (nibble)  {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) ((data>>24) & 0x0000000F);
    if (nibble || noLead) {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) ((data>>20) & 0x0000000F);
    if (nibble || noLead) {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) ((data>>16) & 0x0000000F);
     if (nibble || noLead) {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) ((data>>12) & 0x0000000F);
    if (nibble || noLead) {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) ((data>>8) & 0x0000000F);
    if (nibble || noLead) {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) ((data>>4) & 0x0000000F);
    if (nibble || noLead) {
        sendNibble(nibble);
        noLead = 1;
    }
    nibble = (uint8_t) (data & 0x0000000F);
    sendNibble(nibble);
}


void send0X32(uint32_t data) // **********************************************
// send 1 word (32bit) in hex format WITH leading 0x.Leading zeroes suppressed
// **************************************************************************
{
//	if (data >= 10)  {			//modified 11Jan24
		sendTB('0');
		sendTB('x');
//	}
	sendNX32(data);
}


void sendS32(uint32_t data) // *******************************************
//send signed decimal number string. Leading zeroes are suppressed
//trailing SPACE
//up to  +/- 2.147.483.647 Billions
// **************************************************************************
{
    sendNS32(data);
    sendTB(0x20);
}


 void sendLS32(uint32_t data) // ********************************************
//send signed decimal number string. Leading zeroes are suppressed
//leading SPACE
//up to  +/- 2.147.483.647 Billions
// **************************************************************************
{
    sendTB(0x20);
    sendNS32(data);
}


void sendNS32(uint32_t data) // *******************************************
//send signed decimal number string. Leading zeroes are suppressed
//no leading and trailing SPACE
//up to  +/- 2.147.483.647 Billions
// **************************************************************************
{
uint32_t quotient;

    if (data >= 0x80000000) {
        sendTB('-');
        data = ~data + 1 ;
    }
    if (data < 10) goto ONES;
    if (data < 100) goto TENS;
    if (data < 1000) goto HUNDREDS;
    if (data < 10000) goto THOUSANDS;
    if (data < 100000) goto TH10;
    if (data < 1000000) goto TH100;
    if (data < 10000000) goto MILLIONS;
    if (data < 100000000) goto MIL10;
    if (data < 1000000000) goto MIL100;
    quotient = data/1000000000;
    data = data - quotient*1000000000;
    sendTB((uint8_t)quotient+ 0x30);
MIL100:
    quotient = data/100000000;
    data = data - quotient*100000000;
    sendTB((uint8_t)quotient + 0x30);
MIL10:
    quotient = data/10000000;
    data = data - quotient*10000000;
    sendTB((uint8_t)quotient + 0x30);
MILLIONS:
    quotient = data/1000000;
    data = data - quotient*1000000;
    sendTB((uint8_t)quotient + 0x30);
TH100:
    quotient = data/100000;
    data = data - quotient*100000;
    sendTB((uint8_t)quotient+ 0x30);
TH10:
    quotient = data/10000;
    data = data - quotient*10000;
    sendTB((uint8_t)quotient + 0x30);
THOUSANDS:
    quotient = data/1000;
    data = data - quotient*1000;
    sendTB((uint8_t)quotient + 0x30);
HUNDREDS:
    quotient = data/100;
    data = data - quotient*100;
    sendTB((uint8_t)quotient+ 0x30);
TENS:
    quotient = data/10;
    data = data - quotient*10;
    sendTB((uint8_t)quotient + 0x30);
ONES:
    sendTB((uint8_t)data + 0x30) ;
}


uint32_t pullUsbRxBuf(void) // ********************************************
// pull 1 byte asynchronously from USART reveive buffer
// check first if something received. If yes then set bit 15. If not return 0
// less overhead and more safe than use of a separate "buffer filled" flag
// **************************************************************************
{
uint32_t recv;

    servSystem();   //serve LED, watchdog etc.

    if (gSTLINK_RXTAIL!=gSTLINK_RXHEAD)  {
        recv= (uint32_t)((*gSTLINK_RXTAIL++ & 0x000000FF)|0x00008000);
        if (gSTLINK_RXTAIL==gSTLINK_RXBUFTOP) {
                gSTLINK_RXTAIL=gSTLINK_RXBUF;
        }
		LED_Off();
        return recv;
    }
    else return 0;
}

#ifndef USARTIO
uint32_t pullTermRxBuf(void)  // ********************************************
// receive byte from serial terminal port with priority order USB > RS232
// **************************************************************************
{
	return pullUsbRxBuf();
}

#else

uint32_t pullUsartRxBuf(void) // ********************************************
// pull one byte asynchronously from USART reveive buffer
// check first if something received. If yes then set bit 15. If not return 0
// less overhead and more safe than use of a separate "buffer filled" flag
// **************************************************************************
{
uint32_t recv;

	servSystem();   //serve watchdog etc.

	if (gUSART_RXTAIL != gUSART_RXHEAD)  {
		recv= (uint32_t)((*gUSART_RXTAIL++ & 0x000000FF)|0x00008000);
		if (gUSART_RXTAIL >= gUSART_RXBUFTOP)  gUSART_RXTAIL=gUSART_RXBUF;
		LED_Off();
		return recv;
	}
	else return 0;
}


uint32_t pullTermRxBuf(void)  // ********************************************
// receive byte from serial terminal port with priority order USB > RS232
// **************************************************************************
{
uint32_t recv;

    if (gSplit == 0)  {
		recv = pullUsbRxBuf();
		if (recv) return recv;	//USB has priority over USART
		recv = pullUsartRxBuf();
		return recv;
    }
    else  {						//gSplit = 1 or 3: USB is terminal
        recv = pullUsbRxBuf();
        return recv;
    }
}


void  MidiScanner(uint32_t byte)  // ****************************************
// accepts any MIDI channel (differs from G4xx version)
// **************************************************************************
{
static uint32_t status, data1, msgtype;
static uint32_t progress;

    if (byte >= 0x80) {					//new status byte
			if ((byte == 0xF0) || (byte == EOX) || (byte == 0xF4) ||
						(byte == 0xF5) || (byte == 0xF9) ||(byte == 0xFD) ) {
//								kill SysEx(F0),EOX is end of SysEx message,
//								0xF4, 0xF5, 0xF9 and 0xFD are not defined
				progress = 0;			//reset parser
				status = 0;				//drop input until next status byte
				msgtype = 0;
				return;
			}
			if ((byte == 0xF6) || (byte >= 0xF8))  { //1 byte msg
				gMIDIBUF[gMIDI_HEAD] = byte << 16;		//data bytes = 0
				gMIDI_HEAD++;
				if (gMIDI_HEAD == MIDI_BUFLEN) gMIDI_HEAD = 0;
				progress = 0;			//reset parser
				status = 0;				//drop input until next status byte
				msgtype = 0;
				return;
			}
// "normal" MIDI Channel Msg:
			status = byte;
			msgtype = byte & 0xF0;
			progress = 1;
	}
	else  {								//new data byte
		if (progress == 0)  {	//SysEx or byte out of context, reset parser
			status = 0;
			msgtype = 0;
			data1 = 0;
			return;
		}
		if (progress == 1)  {
			data1 = byte;
			if ((msgtype == 0xC0)||(msgtype == 0xD0) || (status == 0xF1)||
									(status == 0xF3)) { //2 byte msgs
				gMIDIBUF[gMIDI_HEAD] = (status << 16) | data1;
				gMIDI_HEAD++;					//bits 0..7 for easier parsing
				if (gMIDI_HEAD == MIDI_BUFLEN) gMIDI_HEAD = 0;
				return;		//keep progress==1 and gStatus for Running State
			}
			progress = 2;
			return;
		}
		if (progress == 2)  {					//any 3 byte messages
			gMIDIBUF[gMIDI_HEAD] = (status << 16) | (data1 << 8) | byte;
			gMIDI_HEAD++;					//bits 0..7 for easier parsing
			if (gMIDI_HEAD == MIDI_BUFLEN) gMIDI_HEAD = 0;
			progress = 1;   //reset progress and keep status for Running State
		}
	}
}


uint32_t pullMidiBuf(void) // ********************************************
// pull one word asynchronously from MIDIBUF
// check first if something received. If yes then set bit 31. If not return 0
// less overhead and more safe than use of a separate "buffer filled" flag
// **************************************************************************
{
uint32_t recv;

	servSystem();   //serve watchdog etc.

	if (gMIDI_TAIL != gMIDI_HEAD)  {
		recv= gMIDIBUF[gMIDI_TAIL] | 0x80000000;
		gMIDI_TAIL++;
		if (gMIDI_TAIL== MIDI_BUFLEN)  gMIDI_TAIL= 0;
		LED_Off();
		return recv;
	}
	else return 0;
}
#endif


// Interrupt Handlers
//*******************

void SysTick_Handler(void)  // ***********************************************
// ***************************************************************************
{
	if (gSYSCOUNT) gSYSCOUNT--;			//controls servSystem()
	if (gDELAYCOUNT) gDELAYCOUNT--;
	if (gTIMEOUT) gTIMEOUT--;			//to be used in any subroutine
	if (gfMSCOUNT) gfMSCOUNT --;		//used by MS, DS
	if (gfTIXCOUNT) gfTIXCOUNT --;		//used by  TIX, WAIT
	if (gfSTEPCOUNT > 0) gfSTEPCOUNT--;	//stepper timing
	gfRandom ^= gfRandom << 13;			//xorshift32 algorithm/white noise
	gfRandom ^= gfRandom >> 17;
	gfRandom ^= gfRandom << 5;
}

void USART2_IRQHandler(void)    // *****************************************
// *************************************************************************
{
uint32_t recv;

    if (USART2->ISR & USART_ISR_RXNE)  {
		recv = USART2->RDR & 0x00FF;

#ifdef USARTIO
		if (gTransparent)  {		//double if saves runtime in std.case
			if (recv == CTRL_T)  {
				gTransEnd++;
				if (gTransEnd >= 3)  {
					gTransEnd = 0;
					gTransparent = 0;
				}
				return;
			}
			else  {
				while (gTransEnd)  {	//re-feed suppressed CTRL_T data
					*gSTLINK_RXHEAD = CTRL_T;
					gSTLINK_RXHEAD++;
					if (gSTLINK_RXHEAD==gSTLINK_RXBUFTOP)  {
						gSTLINK_RXHEAD=gSTLINK_RXBUF;
					}
					gTransEnd--;
				}		
				gTransEnd = 0;			//precaution, should already be zero
			}
			*gSTLINK_RXHEAD = (uint8_t)recv;
			gSTLINK_RXHEAD++;
			if (gSTLINK_RXHEAD==gSTLINK_RXBUFTOP)  gSTLINK_RXHEAD=gSTLINK_RXBUF;
			return;
		}
#endif
		if (recv == CTRL_C)  {
			gfTermInput = CTRL_C;
			return;
		}
		if (recv != CTRL_B)  {
			*gSTLINK_RXHEAD = (uint8_t)recv;
			gSTLINK_RXHEAD++;
			if (gSTLINK_RXHEAD==gSTLINK_RXBUFTOP)  gSTLINK_RXHEAD=gSTLINK_RXBUF;
		}
		else  op_BAKOP();
    }
}

#ifdef USARTIO
void USART1_IRQHandler(void)    // *****************************************
// *************************************************************************
{
uint32_t recv;

	if (USART1->ISR & USART_ISR_RXNE)  {
		recv = USART1->RDR & 0x00FF;
		if (gSplit == 0) {	//	USART is terminal
			if (recv == CTRL_C)  {
				gfTermInput = CTRL_C;
				return;
			}
			if (recv != CTRL_B)  {
				*gUSART_RXHEAD = (uint8_t)recv;
				gUSART_RXHEAD++;
				if (gUSART_RXHEAD==gUSART_RXBUFTOP) gUSART_RXHEAD=gUSART_RXBUF;
			}
			else  op_BAKOP();
		}
		else {									//no special char handling
			*gUSART_RXHEAD = (uint8_t)recv;
			gUSART_RXHEAD++;
			if (gUSART_RXHEAD==gUSART_RXBUFTOP)  gUSART_RXHEAD=gUSART_RXBUF;
		}
	}
}
#endif

// #########################################################################
